import os
#import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxStringParameter, SandboxBoolParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.paths import get_logs_folder

from sandbox.projects import resource_types
from sandbox.projects.common.utils import get_or_default


class PublishSimilarOrgs(SandboxTask):
    """
        Calculates geosearch similar orgs from hypotheses
    """
    type = 'PUBLISH_SIMILAR_ORGS'

    class ExportPreparer(LastReleasedResource):
        name = 'export_preparer'
        description = 'Export preparer: '
        resource_type = resource_types.CREATE_SIMILAR_ORGS_EXPORT_EXECUTABLE

    class SimilarOrgs(LastReleasedResource):
        name = 'similar_orgs'
        description = 'Similar orgs: '
        resource_type = resource_types.SIMILAR_ORGS_LIST

    class BusinessIndex(LastReleasedResource):
        name = 'business_index'
        description = 'Business index: '
        resource_type = resource_types.MAPS_DATABASE_BUSINESS

    class PhotosExport(LastReleasedResource):
        name = 'photos_export'
        description = 'Photos export: '
        resource_type = resource_types.BUSINESS_SNIPPET_PHOTOS

    class SaasDumper(LastReleasedResource):
        name = 'saas_dumper'
        description = 'SaaS dumper: '
        resource_type = resource_types.DUMP_TO_INDEXERPROXY_EXECUTABLE

    class MrCluster(SandboxStringParameter):
        description = 'YT cluster'
        name = 'cluster'
        default_value = 'hahn.yt.yandex.net'

    class MrTable(SandboxStringParameter):
        description = 'Table for similar orgs publishing'
        name = 'table'
        default_value = '//home/geosearch/similar_orgs/saas_export'

    class SaasIndexer(SandboxStringParameter):
        description = "SaaS indexer"
        name = 'saas_indexer'
        choices = [
            ('testing', 'saas-indexerproxy-maps-prestable.yandex.net'),
            ('stable', 'saas-indexerproxy-maps-kv.yandex.net'),
        ]
        default_value = 'saas-indexerproxy-maps-prestable.yandex.net'

    class ExperimentPrefix(SandboxStringParameter):
        description = 'Experiment tag'
        name = 'experiment_prefix'
        default_value = ''

    class DumpToSaasOnly(SandboxBoolParameter):
        description = 'Dump to saas only'
        name = 'to_saas_only'
        hidden = False
        default_value = False

    input_parameters = (
        ExportPreparer,
        SimilarOrgs,
        BusinessIndex,
        PhotosExport,
        SaasDumper,
        MrCluster,
        MrTable,
        SaasIndexer,
        ExperimentPrefix,
        DumpToSaasOnly,
    )

    environment = (environments.PipEnvironment('yandex-yt'),)

    @staticmethod
    def upload_export(token, cluster, table, export, prefix):
        import yt.wrapper
        from yt.wrapper import YtClient
        yt_line = PublishSimilarOrgs.yt_line_from_input
        client = YtClient(proxy=cluster, token=token)
        tmp_table = table + '.tmp'
        client.write_table(tmp_table, (yt_line(line, prefix) for line in export if len(yt_line(line, prefix))), format=yt.wrapper.JsonFormat(attributes={"encode_utf8": False}))
        client.move(tmp_table, table, force=True)

    @staticmethod
    def yt_line_from_input(inp, exp_prefix):
        result = {}
        split = inp.rstrip('\n').split('\t')
        if len(split) < 3:
            return result
        data = split[2] if len(split) == 3 else ' '.join(split[2:])
        key_fields = ['sim', split[0], split[1]]
        if exp_prefix:
            key_fields.append(exp_prefix)
        return {
            'key': '~'.join(key_fields),
            'value': 'similars/1.x=' + data
        }

    def on_execute(self):

        calcer = self.sync_resource(self.ctx.get('export_preparer'))
        similar_orgs = self.sync_resource(self.ctx.get('similar_orgs'))
        index = self.sync_resource(self.ctx.get('business_index'))
        photos = self.sync_resource(self.ctx.get('photos_export'))
        dumper = self.sync_resource(self.ctx.get('saas_dumper'))
        cluster = get_or_default(self.ctx, self.MrCluster)
        table = get_or_default(self.ctx, self.MrTable)
        exp_prefix = get_or_default(self.ctx, self.ExperimentPrefix)
        indexer = get_or_default(self.ctx, self.SaasIndexer)
        dump_only = get_or_default(self.ctx, self.DumpToSaasOnly)
        export = './similar_orgs_export.txt'

        yt_token = self.get_vault_data('mesherin', 'YQL_TOKEN')
        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = yt_token

        if not dump_only:
            exec_params = [calcer, os.path.join(index, 'companies.mms'), photos]
            with open(export, 'w') as output:
                run_process(exec_params, stdout=output, stdin=open(similar_orgs))

            self.upload_export(yt_token, cluster, table, open(export), exp_prefix)

        upload_params = [dumper,
            '--mode', 'mr',
            '-k', '125eacfe0f6b45c6a7476d7575527d72',
            '-h', indexer,
            '--mr-server', cluster,
            '--mr-input', table,
            '--mr-output', table + '.err',
            '--mr-processor', 'personal',
            '--prefix', '1',
            '-l', '4',
        ]
        dump_err = os.path.join(get_logs_folder(), 'indexerproxy_dumper.err')
        try:
            run_process(upload_params, stderr=open(dump_err, 'w'))
        except Exception as e:
            self.create_resource('dump to indexer proxy log', dump_err, resource_types.OTHER_RESOURCE)
            raise e


__Task__ = PublishSimilarOrgs
