# coding: utf-8

import logging
import tarfile
import os.path
import re

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxSvnUrlParameter, SandboxBoolParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

DEFAULT_FORMULA = 'formulas/matrixnet.info'
DIR = 'formulas'


class FormulasSvnUrlParameter(SandboxSvnUrlParameter):
    name = 'svn_url'
    description = 'SVN full path to formulas (e.g. svn+ssh://arcadia.yandex.ru/arc/tags/antirobot-formulas/rXX/)'


class MustExistMatrixnetInfoParametr(SandboxBoolParameter):
    name = 'check_matrixnet_info'
    description = 'Must exist file: %s' % DEFAULT_FORMULA
    default_value = True


class ReleaseAntirobotFormulas(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Выкачиваем АнтиРоботные формулы из svn, пакуем в tar, сохраняем как ресурс
    """

    type = 'RELEASE_ANTIROBOT_FORMULAS'

    input_parameters = (
        FormulasSvnUrlParameter,
        MustExistMatrixnetInfoParametr,
    )

    def on_execute(self):
        revision, tag, branch = self.arcadia_info()
        subPath = tag if tag is not None else branch

        formulas_path = self.path(DIR)
        logging.info('Fetching formulas')
        svn_url_info = Arcadia.info(self.ctx['svn_url'])
        Arcadia.export(svn_url_info['url'], formulas_path, revision)

        with open(self.path(DIR + '/svn_info.txt'), 'wt') as svn_info:
            print >> svn_info, svn_url_info

        formulas_tgz_path = self.path('formulas.tar.gz')
        logging.info('Pack to .tgz')
        with tarfile.open(formulas_tgz_path, 'w:gz') as tar:
            tar.add(formulas_path, arcname='formulas')

        if self.ctx['check_matrixnet_info']:
            logging.info('Lookup %s' % DEFAULT_FORMULA)
            if not os.path.isfile(self.path(DEFAULT_FORMULA)):
                logging.error("Couldn't find %s. Exit" % DEFAULT_FORMULA)
                raise SandboxTaskFailureError("Couldn't find %s " % DEFAULT_FORMULA)

        logging.info('Creating resource')
        self.create_resource('antirobot formulas %s r%s' % (subPath, revision), formulas_tgz_path, resource_types.ANTIROBOT_FORMULAS, arch='any')
        logging.info('Done!')

    def arcadia_info(self):
        path = self.ctx['svn_url']
        parsed_url = Arcadia.parse_url(path)
        revision = parsed_url.revision if parsed_url.revision is not None else Arcadia.info(path)['entry_revision']
        subPath = ''
        try:
            subPath = re.search(r'.*arc/(tags|branches)/([^@]*)', path).group(2).rstrip('/')
        except:
            raise SandboxTaskFailureError('Tags or Branches regexp failure for: %s' % path)
        if 'arc/tags' in path:
            tag = subPath
            branch = None
        elif 'arc/branches' in path:
            tag = None
            branch = subPath + '@' + str(revision)
        else:
            raise SandboxTaskFailureError('Unknown tag or branch: %s' % path)
        return revision, tag, branch


__Task__ = ReleaseAntirobotFormulas
