# coding: utf-8

# $Header$

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask


class ReleaseAny(SandboxTask):
    type = 'RELEASE_ANY'

    description = 'Releaser for arbitrary tasks'
    cores = 1
    required_ram = 1024
    execution_space = 64  # Mb

    def is_resources_already_released(self):
        self.info = "Check is same resources already released in task with same release status:\n"
        curr_rel_resources = channel.sandbox.list_resources(task_id=self.ctx['release_task_id'])
        if not curr_rel_resources:
            self.info += "No resources found for releasing task {}\n".format(self.ctx['release_task_id'])
            return False

        curr_resources = {}
        for r in (curr_rel_resources):
            curr_resources[str(r.type)] = r.file_md5

        prev_task_id = None

        for res in (self.ctx['check_already_released'].split()):  # space separated list

            if not curr_resources.get(res, None):
                self.info += "Appointed task has no md5 for mandatory resource {}\n".format(res)
                raise SandboxTaskFailureError("Appointed task has no md5 for mandatory resource {}".format(res))

            rel = channel.sandbox.list_releases(
                resource_type=res,
                release_status=self.ctx['release_status'],
                limit=1,
                order_by='-id'
            )

            if not rel or not rel[0]:
                self.info += "No releases found for {}\n".format(str(res))
                return False

            if not prev_task_id:
                self.info += "Prevoius release task for resource {} is {}. ".format(res, rel[0].id)
                self.info += "Assume it's a latest release for whole bunch of our resources\n"
                prev_task_id = rel[0].id

            if prev_task_id != rel[0].id:
                self.info += "Release tasks for resources are different.\n"
                self.info += "Latest task for {} is {}\n".format(res, rel[0].id)
                return False

            prev_res_md5 = None

            for r in (rel[0].resources):
                if str(r.type) == res and r.file_md5:
                    self.info += "{} md5: curr:{} vs prev:{}\n".format(res, curr_resources[res], r.file_md5)
                    if curr_resources[res] != r.file_md5:
                        self.info += "\nMandatory resource {} has different md5.\n".format(res)
                        return False
                    prev_res_md5 = r.file_md5

            if not prev_res_md5:
                self.info += "Task {} has no md5 for mandatory resource {}\n".format(prev_task_id, res)
                return False

        self.info += "All resources already released in task {}.\n".format(prev_task_id)
        return True

    def on_execute(self):
        self.info = "Task, appointed to release: {}\n".format(self.ctx['release_task_id'])

        # sometimes (quite often in prod sandbox) releaser try to do it's job before caller completely finished
        task = channel.sandbox.get_task(self.ctx['release_task_id'])
        if not task.is_done():
            self.wait_tasks(
                [self.ctx['release_task_id']],
                (self.Status.Group.FINISH + self.Status.Group.BREAK),
                wait_all=True,
            )

        # check is same resources already released in prev task with same release status
        if self.ctx.get('check_already_released', False):
            if self.is_resources_already_released():
                self.info += "\nRelease cancelled.\n"
                return
            else:
                self.info += "\nData looks fresh. Release!\n"

        # wait previews tasks
        wait_taskes_ids = self.ctx.get('wait_tasks_ids', None)
        if wait_taskes_ids:
            to_wait = []
            for w_id in wait_taskes_ids:
                t = channel.sandbox.get_task(w_id)
                if not t.is_done():
                    to_wait.append(w_id)
            if to_wait:
                self.wait_tasks(
                    to_wait,
                    (self.Status.Group.FINISH + self.Status.Group.BREAK),
                    wait_all=True,
                )

        # release at least =)
        self.create_release(
            self.ctx['release_task_id'],
            status=self.ctx.get('release_status', 'cancelled'),
            subject=self.ctx.get('release_subject', None),
            comments=self.ctx.get('release_comments', None),
            changelog=self.ctx.get('release_changelog', None),
            addresses_to=self.ctx.get('release_addresses_to', None),
            addresses_cc=self.ctx.get('release_addresses_cc', None),
            additional_parameters=self.ctx.get('release_additional_parameters', None)
        )


__Task__ = ReleaseAny
