# -*- coding: utf-8 -*-

from sandbox.common.types.task import Status
from sandbox.projects import resource_types
from sandbox.projects.BuildDelayedViewEntityBaseTrie import BuildDelayedViewEntityBaseTrie
from sandbox.projects.BuildDelayedViewEntityBaseTrie.parameters import EmptyEntityBaseTrie
from sandbox.projects.BuildDelayedViewSerialBaseTrie import BuildDelayedViewSerialBaseTrie
from sandbox.projects.BuildDelayedViewSerialBaseTrie.parameters import EmptySerialBaseTrie

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxSvnTemporaryError
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.svn import Arcadia
import datetime
import os
import sandbox.projects.BuildDelayedViewEntityBaseTrie.parameters as entity_task_parameters
import sandbox.projects.BuildDelayedViewSerialBaseTrie.parameters as serial_task_parameters
import sandbox.projects.video.quality.delayed_view as delayed_view
from sandbox import sandboxsdk
import time


class DelayedViewRearrangeDynamicYaMakeArcadiaPath(SandboxStringParameter):
    description = 'Arcadia trunk path to delayed view rearrange.dynamic ya make'
    name = 'delayed_view_rearrange_dynamic_ya_make_arcadia_path'
    required = True


def get_temp_yt_table_path(yt_proxy, yt_token):
    import yt.wrapper as yt
    yt_client = yt.YtClient(yt_proxy, yt_token)
    expiration_timeout_ms = datetime.timedelta(days=1).total_seconds() * 1000.0
    return yt_client.create_temp_table(expiration_timeout=expiration_timeout_ms)


def get_entity_main_base_yt_table_path(version='latest'):
    return '//home/dict/ontodb/ver/daily/{}/all_cards_final'.format(version)


def get_entity_delta_base_yt_table_path(version='latest'):
    return '//home/dict/ontodb/ver/main_delta_km/{}/all_cards_final'.format(version)


class ReleaseDelayedViewTries(sandboxsdk.task.SandboxTask):
    '''Release tries for delayed view in video-search'''

    type = 'RELEASE_DELAYED_VIEW_TRIES'

    input_parameters = delayed_view.get_common_params() + [
        DelayedViewRearrangeDynamicYaMakeArcadiaPath, EmptyEntityBaseTrie, EmptySerialBaseTrie]

    environment = (
        environments.PipEnvironment('yandex-yt'),
    )

    BUILD_ENTITY_TRIE_TASK_ID = 'build_entity_trie_task_id'
    BUILD_SERIAL_TRIE_TASK_ID = 'build_serial_trie_task_id'

    ENTITY_TRIE_RESOURCE_ID = 'entity_trie_resource_id'
    SERIAL_TRIE_RESOURCE_ID = 'serial_trie_resource_id'

    ENTITY_TRIE_RESOURCE_PATH = 'entity_trie_resource_path'
    SERIAL_TRIE_RESOURCE_PATH = 'serial_trie_resource_path'

    ROBOT_NAME = 'robot-video-pers'
    ROBOT_PRIVATE_KEY_VAULT_NAME = 'ssh_key'

    def __save_entity_trie_resource(self, resource):
        self.ctx[self.ENTITY_TRIE_RESOURCE_ID] = resource.id
        self.ctx[self.ENTITY_TRIE_RESOURCE_PATH] = os.path.basename(resource.path)

    def __save_serial_trie_resource(self, resource):
        self.ctx[self.SERIAL_TRIE_RESOURCE_ID] = resource.id
        self.ctx[self.SERIAL_TRIE_RESOURCE_PATH] = os.path.basename(resource.path)

    def __build_entity_trie_stage(self):
        yt_proxy, yt_token = delayed_view.get_yt_params(self)

        # common and EmptyEntityBaseTrie params are set implicitly
        self.ctx[entity_task_parameters.EntityMainBaseYtTablePath.name]\
            = get_entity_main_base_yt_table_path()
        self.ctx[entity_task_parameters.EntityDeltaBaseYtTablePath.name]\
            = get_entity_delta_base_yt_table_path()
        self.ctx[entity_task_parameters.DelayedViewEntityBaseYtTablePath.name]\
            = get_temp_yt_table_path(yt_proxy, yt_token)
        self.ctx[entity_task_parameters.UseExistingEntityBaseYtTable.name] = False

        build_entity_trie_task = self.create_subtask(
            task_type=BuildDelayedViewEntityBaseTrie.type,
            description='Release delayed view tries')
        self.ctx[self.BUILD_ENTITY_TRIE_TASK_ID] = build_entity_trie_task.id

        self.wait_tasks(build_entity_trie_task, Status.Group.FINISH, wait_all=True)

    def __extract_entity_trie_resource_stage(self):
        resource = delayed_view.get_resource(self.ctx[self.BUILD_ENTITY_TRIE_TASK_ID], resource_types.DELAYED_VIEW_ENTITY_BASE_TRIE)
        self.__save_entity_trie_resource(resource)

    def __build_serial_trie_stage(self):
        # common and EmptySerialBaseTrie params are set implicitly
        self.ctx[serial_task_parameters.DelayedViewSerialBaseYtTablePath.name]\
            = serial_task_parameters.DelayedViewSerialBaseYtTablePath.default_value

        build_serial_trie_task = self.create_subtask(
            task_type=BuildDelayedViewSerialBaseTrie.type,
            description='Release delayed view tries')
        self.ctx[self.BUILD_SERIAL_TRIE_TASK_ID] = build_serial_trie_task.id

        self.wait_tasks(build_serial_trie_task, Status.Group.FINISH, wait_all=True)

    def __extract_serial_trie_resource_stage(self):
        resource = delayed_view.get_resource(self.ctx[self.BUILD_SERIAL_TRIE_TASK_ID], resource_types.DELAYED_VIEW_SERIAL_BASE_TRIE)
        self.__save_serial_trie_resource(resource)

    def __get_rearrange_dynamic_ya_make_content(self):
        result = ['OWNER(', '    g:video-quality', ')', '', 'UNION()', '']
        result += ['FROM_SANDBOX(FILE {} OUT_NOAUTO {})'.format(self.ctx[self.ENTITY_TRIE_RESOURCE_ID], self.ctx[self.ENTITY_TRIE_RESOURCE_PATH])]
        result += ['FROM_SANDBOX(FILE {} OUT_NOAUTO {})'.format(self.ctx[self.SERIAL_TRIE_RESOURCE_ID], self.ctx[self.SERIAL_TRIE_RESOURCE_PATH])]
        result += ['', 'END()']
        return '\n'.join(result)

    def __save_tries_to_rearrange_dynamic_stage(self):
        rd_relative_path = self.ctx[DelayedViewRearrangeDynamicYaMakeArcadiaPath.name]

        working_copy_dir = self.path('working_copy')
        os.mkdir(working_copy_dir)
        Arcadia.checkout(
            os.path.join(Arcadia.trunk_url(), os.path.dirname(rd_relative_path)),
            working_copy_dir, depth='empty')
        Arcadia.update(working_copy_dir, set_depth='files')

        rd_absolute_path = os.path.join(working_copy_dir, os.path.basename(rd_relative_path))

        if not os.path.exists(rd_absolute_path):
            raise SandboxTaskFailureError(
                'Cannot find ya make file in delayed view rearrange dynamic.'
                'Release task is supposed to update ya make file but not create it')

        with open(rd_absolute_path, 'w') as rearrange_dynamic_ya_make_file:
            result = self.__get_rearrange_dynamic_ya_make_content()
            rearrange_dynamic_ya_make_file.write(result + '\n')

        with ssh.Key(self, self.ROBOT_NAME, self.ROBOT_PRIVATE_KEY_VAULT_NAME):
            self.ctx['svn_retries'] = 0
            while True:
                try:
                    commit_msg = '[AUTO_COMMIT] '
                    commit_msg += '[diff-resolver:vvp] '
                    commit_msg += 'Add new version of delayed view tries'
                    Arcadia.update(working_copy_dir, set_depth='files')
                    Arcadia.commit(rd_absolute_path, message=commit_msg, user=self.ROBOT_NAME)
                except SandboxSvnTemporaryError:
                    if self.ctx['svn_retries'] == 10:
                        raise SandboxTaskFailureError('Cannot commit to rearrange dynamic ya.make')
                    self.ctx['svn_retries'] += 1
                    time.sleep(30)
                else:
                    break

    def on_execute(self):
        stage_funcs = [self.__build_entity_trie_stage,
                       self.__extract_entity_trie_resource_stage,
                       self.__build_serial_trie_stage,
                       self.__extract_serial_trie_resource_stage,
                       self.__save_tries_to_rearrange_dynamic_stage]

        delayed_view.process_stages(self, stage_funcs)


__Task__ = ReleaseDelayedViewTries
