# -*- coding: utf-8 -*-


import re
import os
import logging
from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxArcadiaUrlParameter, SandboxBoolParameter
from sandbox.projects.common.nanny import nanny


logger = logging.getLogger(__name__)


class ShardmapResourceType(SandboxStringParameter):
    name = 'shardmap_resource_type'
    description = 'Shardmap resource type'
    choices = [(str(x), str(x)) for x in resource_types.__dict__]
    required = True


class ArcadiaSvnShardDir(SandboxArcadiaUrlParameter):
    name = 'svn_shardmap_dir'
    description = 'Arcadia svn shards dir'
    required = True


class ReleaseLatest(SandboxBoolParameter):
    name = 'release_latest'
    description = 'Release only newest shardmap'
    default_value = False


class ShardmapNameRegexp(SandboxStringParameter):
    name = 'shardmap_regexp'
    description = 'Shardmap name regexp'
    default_value = 'shardmap-(?P<shardmap_ts>\d+)-(?P<shardmap_date>\d+)-(?P<shardmap_time>\d+).map'
    required = True

    @classmethod
    def cast(cls, value):
        re.compile(value)
        return value


class ReleaseStatus(SandboxStringParameter):
    name = 'release_status'
    description = 'Release shardmap with status'
    choices = [
        ('', False),
        ('stable', 'stable'),
        ('prestable', 'prestable'),
        ('testing', 'testing'),
        ('unstable', 'unstable'),
    ]
    required = False


class ShardmapNamePrefix(SandboxStringParameter):
    name = 'shardmap_name_prefix'
    description = 'Shardmap name prefix'
    required = False


class ReleaseMediaShardmaps(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Находит новые шардмапы в SVN и релизит их
    """

    type = 'RELEASE_MEDIA_SHARDMAPS'
    execution_space = 256
    input_parameters = [ShardmapResourceType,
                        ArcadiaSvnShardDir,
                        ShardmapNameRegexp,
                        ReleaseStatus,
                        ShardmapNamePrefix]

    def local_init(self):
        """Заглушка для локального объявления параметров при наследовании класса"""
        pass

    def postaction(self):
        """Заглушка для действия после релиза шардмапа в наследуемых классах"""
        pass

    def on_execute(self):
        self.local_init()
        shardmap_reg = re.compile(self.ctx['shardmap_regexp'])
        shardmap_service = os.path.basename(os.path.normpath(self.ctx['svn_shardmap_dir']))
        if 'shardmap_name_prefix' in self.ctx and self.ctx['shardmap_name_prefix'] != '':
            shardmap_service = self.ctx['shardmap_name_prefix']

        resources = channel.sandbox.list_resources(resource_type=self.ctx['shardmap_resource_type'],
                                                   status='READY',
                                                   limit=999)

        # Find already published resources
        released_shardmaps = []
        if resources:
            for resource in resources:
                if 'shardmap_name' in resource.attributes:
                    released_shardmaps.append(resource.attributes['shardmap_name'])
                else:
                    logging.debug("Wrong resource: %s" % resource)
        logger.debug('Got already published shardmaps: {}'.format(released_shardmaps))

        checkout_folder = make_folder('sandboxsvn_%s' % shardmap_service)
        Arcadia.checkout(self.ctx['svn_shardmap_dir'], path=checkout_folder, depth='files')

        svn_shardmaps = sorted(os.listdir(checkout_folder))
        if self.ctx.get('release_latest', False):
            logging.debug("Checking latest shadmap: {}".format)
            svn_shardmaps = svn_shardmaps[-1:-2:-1]

        for shardmap in svn_shardmaps:
            m = shardmap_reg.match(shardmap)
            if m:
                self.ctx['shardmap_full_name'] = "%s_%s" % (shardmap_service, shardmap)
                self.ctx['shardmap_file_name'] = shardmap
                self.ctx['shardmap_ts'] = m.group('shardmap_ts')
                self.ctx['shardmap_date'] = m.group('shardmap_date')
                self.ctx['shardmap_time'] = m.group('shardmap_time')

                if self.ctx['shardmap_full_name'] not in released_shardmaps:
                    self.released_new = True
                    logger.debug('Publishing new resource: %s' % self.ctx['shardmap_full_name'])
                    resource_class = getattr(resource_types, self.ctx['shardmap_resource_type'])
                    self.ctx['shardmap_resource'] = self.create_resource(
                        shardmap,
                        os.path.join(checkout_folder, shardmap),
                        resource_class,
                        attributes={'shardmap_name': self.ctx['shardmap_full_name'],
                                    'shardmap_timestamp': self.ctx['shardmap_ts']}).id

                    # We must release only one resource for each type for one task run
                    if 'release_status' in self.ctx and self.ctx['release_status']:
                        self.create_subtask('RELEASE_PARENT_TASK', self.ctx['release_status'])
                    break
                else:
                    logger.debug('Resource with name %s already published' % self.ctx['shardmap_full_name'])
            else:
                logger.debug('Incorrect file name: %s' % shardmap)

        self.postaction()


__Task__ = ReleaseMediaShardmaps
