import copy
import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxBoolParameter

from sandbox.projects.common.apihelpers import get_last_released_resource

from sandbox.common.types.task import Status

from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_IEX_PROXY
from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_MSEARCH_PROXY
from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_LUCENE
from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_PERSEUS
from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_ZOOLOSER
from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_YADISK_SEARCH_PROXY
from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_SALO
from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_TUPITA
from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_LOGBROKER_CONSUMER
from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_ZORA_PROXY
from sandbox.projects.MakePersonalBundle import PERSONAL_SEARCH_DAEMON_LUCENE_MAIL_CONFIG

from sandbox.projects.MakePersonalBundle import RepoUrl
from sandbox.projects.MakePersonalBundle import ProjectNames
from sandbox.projects.MakePersonalBundle import JavaResourceId
from sandbox.projects.MakePersonalBundle import RagelResourceId
from sandbox.projects.MakePersonalBundle import ZlibsResourceId
from sandbox.projects.MakePersonalBundle import SqliteResourceId
from sandbox.projects.MakePersonalBundle import GccResourceId
from sandbox.projects.MakePersonalBundle import GenerateSonarReport
from sandbox.projects.MakePersonalBundle import SonarScannerResourceId
from sandbox.projects.MakePersonalBundle import NumberOfTries
from sandbox.projects.MakePersonalBundle import NumberOfThreads
from sandbox.projects.MakePersonalBundle import AddAllToMake

from sandbox.sandboxsdk import environments


class UpdateAnyway(SandboxBoolParameter):
    name = 'update_anyway'
    description = 'Set to True if make you want to make new resources anyway'
    required = False
    default_value = False


class ReleasePersonalBundle(SandboxTask):
    """
        Runs subtask that makes new sandbox resource with bundle for specified
        project from git repo and then releases that task. For Nanny integration
    """

    type = 'RELEASE_PERSONAL_BUNDLE'

    execution_space = 10000
    environment = (environments.PipEnvironment('GitPython'),)

    input_parameters = [
        RepoUrl,
        ProjectNames,
        JavaResourceId,
        RagelResourceId,
        ZlibsResourceId,
        SqliteResourceId,
        GccResourceId,
        GenerateSonarReport,
        SonarScannerResourceId,
        NumberOfTries,
        NumberOfThreads,
        UpdateAnyway,
        AddAllToMake
    ]

    def on_execute(self):
        import git
        need_update = True
        projects_to_resources = {}
        projects_to_resources['iex-proxy'] = PERSONAL_SEARCH_DAEMON_IEX_PROXY
        projects_to_resources['msearch-proxy'] = PERSONAL_SEARCH_DAEMON_MSEARCH_PROXY
        projects_to_resources['lucene'] = PERSONAL_SEARCH_DAEMON_LUCENE
        projects_to_resources['perseus'] = PERSONAL_SEARCH_DAEMON_PERSEUS
        projects_to_resources['zooloser'] = PERSONAL_SEARCH_DAEMON_ZOOLOSER
        projects_to_resources['yadisk-search-proxy'] = PERSONAL_SEARCH_DAEMON_YADISK_SEARCH_PROXY
        projects_to_resources['salo'] = PERSONAL_SEARCH_DAEMON_SALO
        projects_to_resources['tupita'] = PERSONAL_SEARCH_DAEMON_TUPITA
        projects_to_resources['logbroker-consumer'] = PERSONAL_SEARCH_DAEMON_LOGBROKER_CONSUMER
        projects_to_resources['lucene-mail-config'] = PERSONAL_SEARCH_DAEMON_LUCENE_MAIL_CONFIG
        projects_to_resources['zora-proxy'] = PERSONAL_SEARCH_DAEMON_ZORA_PROXY
        first_project_name = self.ctx.get(ProjectNames.name)[0]
        first_project_resource = projects_to_resources[first_project_name]
        last_resource = get_last_released_resource(first_project_resource, release_status='testing')

        remote_refs = {}
        g = git.cmd.Git()
        for ref in g.ls_remote(self.ctx[RepoUrl.name]).split('\n'):
            logging.info(ref)
            hash_ref_list = ref.split('\t')
            remote_refs[hash_ref_list[1]] = hash_ref_list[0]
        new_sha = str(remote_refs['HEAD'])
        logging.info("new sha = " + new_sha)

        if last_resource is not None:
            logging.info("last_resource description: " + last_resource.description)
            try:
                old_sha = last_resource.description.split("\n")[1].split()[1]
            except:
                old_sha = str(1)
            logging.info('old_sha = ' + old_sha)

            need_update = not new_sha == old_sha
            logging.info("need update = " + str(need_update))

        else:
            logging.info("last_resource is None")
            need_update = True

        if need_update or self.ctx.get(UpdateAnyway.name):
            logging.info("Will clone, make and build new bundle")
            projects_string = ""
            for project_name in self.ctx.get(ProjectNames.name):
                projects_string += project_name + " "
            if 'release_task_id' not in self.ctx:
                coverage_input_params = copy.deepcopy(self.ctx)
                coverage_input_params[ProjectNames.name] = ''
                coverage_input_params[GenerateSonarReport.name] = True
                self.create_subtask(
                    'MAKE_PERSONAL_BUNDLE',
                    description=new_sha + ', sending coverage info to sonar',
                    input_parameters=coverage_input_params
                )

                bundle_input_params = copy.deepcopy(self.ctx)
                bundle_input_params[GenerateSonarReport.name] = False
                subtask = self.create_subtask(
                    'MAKE_PERSONAL_BUNDLE',
                    description=new_sha + ', making bundles for projects ' + projects_string,
                    input_parameters=bundle_input_params
                )
                self.ctx['release_task_id'] = subtask.id
                self.ctx['sha'] = new_sha
                statuses_to_wait = [Status.SUCCESS, Status.FAILURE, Status.DELETED, Status.EXCEPTION, Status.TIMEOUT, Status.STOPPED]
                self.wait_tasks(subtask.id, statuses_to_wait, wait_all=True)
            else:
                logging.info("creating releases")
                release = self.create_release(
                    self.ctx['release_task_id'],
                    status='testing',
                    subject='release for personal projects ' + projects_string + ', ' + self.ctx.get('sha'),
                    comments='comment: testing ' + projects_string
                )

            if not release:
                raise SandboxTaskFailureError('Cannot release task')

        else:
            logging.info("Will not update the resources")


__Task__ = ReleasePersonalBundle
