# coding: utf-8

import shutil
import json

from sandbox.common.errors import TaskFailure
from sandbox.projects.websearch.sdch import SdchReleaseConfig, sdch_releasers
from sandbox import sdk2

from sandbox.projects.common.nanny import nanny


class SdchWebDictionaryPack(sdk2.Resource):
    """
    SDCH dictionaries for WEB, https://wiki.yandex-team.ru/sdch/dictionary-pack/
    """
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = sdch_releasers


class ReleaseSdchDictionaryPack(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
    Release web sdch dictionaries pack https://wiki.yandex-team.ru/sdch/dictionary-pack/
    """

    class Requirements(sdk2.Task.Requirements):
        disk_space = 512

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup("Release phase") as release_phase:
            release_phase.values.stop_using_old_dictionaries = release_phase.Value("stop using old dictionaries")
            release_phase.values.release_new_dictionaries = release_phase.Value("release new dictionaries")
            release_phase.values.start_use_new_dictionaries = release_phase.Value("start use new dictionaries")
        web_dictionary_pack = sdk2.parameters.Resource(
                "SDCH web dictionaries pack",
                required=True,
                resource_type=SdchWebDictionaryPack)

    def on_execute(self):
        if self.Parameters.release_phase == "stop_using_old_dictionaries":
            release_config = sdk2.ResourceData(SdchReleaseConfig(
                                self,
                                "Dictionary release config. Phase 1, stop use old dictionaries. Origin resource: %s" %
                                    self.Parameters.web_dictionary_pack.id,
                                "release_config.json",
                                arch='any'))
            with open(str(release_config.path), "w") as config_file:
                json.dump({"ReleasePhase": "STOP_USING_OLD_DICTIONARY"}, config_file, indent=4)
        elif self.Parameters.release_phase == "release_new_dictionaries":
            original_pack = sdk2.ResourceData(self.Parameters.web_dictionary_pack)
            pack_to_release = sdk2.ResourceData(SdchWebDictionaryPack(
                                        self,
                                        "SDCH dictionaries. Origin resource: %s" %
                                            self.Parameters.web_dictionary_pack.id,
                                        "SDCH.tar",
                                        arch='any',
                                        ttl=180))
            shutil.copyfile(str(original_pack.path), str(pack_to_release.path))
            pack_to_release.ready()

            release_config = sdk2.ResourceData(SdchReleaseConfig(
                        self,
                        "Dictionary release config. Phase 2, release new data. Origin resource: %s" %
                            self.Parameters.web_dictionary_pack.id,
                        "release_config.json",
                        arch='any',
                        ttl=180))
            with open(str(release_config.path), "w") as config_file:
                json.dump({"ReleasePhase": "DO_NOT_USE_NEW_DICTIONARY"}, config_file, indent=4)
            release_config.ready()
        elif self.Parameters.release_phase == "start_use_new_dictionaries":
            release_config = sdk2.ResourceData(SdchReleaseConfig(
                                self,
                                "Dictionary release config. Phase 3, start use new dictionaries. Origin resource: %s" %
                                    self.Parameters.web_dictionary_pack.id,
                                "release_config.json",
                                arch='any'))
            with open(str(release_config.path), "w") as config_file:
                json.dump({"ReleasePhase": "RELEASE_FINISHED"}, config_file, indent=4)
        else:
            raise TaskFailure("Invalid release phase {}".format(self.Parameters.release_phase))
