#!/usr/bin/env python
# -*- coding: utf-8 -*-

from sandbox.common import rest
from sandbox.common.types import resource

from sandbox.projects import resource_types
from sandbox.projects.common.apihelpers import get_last_resource

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxStringParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask

import logging


DEPRECATED_SUFFIX = "TG_DEPRECATED factors not encountered in any formula:"
UNUSED_DEPRECATED = "unused_deprecated_features"


class RankingModelsParameter(LastReleasedResource):
    name = "ranking_models"
    description = "models.archive for checking (last released if empty)"
    resource_type = resource_types.DYNAMIC_MODELS_ARCHIVE
    default_value = ""
    required = False
    do_not_copy = True


class RelevFmlUnusedParameter(SandboxStringParameter):
    name = "relev_fml_unused_params"
    description = "Parameters for relev_fml_unused tool"
    required = True
    default_value = "-u"


class RelevFmlUnused(SandboxTask):
    """
    Task for running relev_fml_unused tool for last released basesearch models
    """

    type = "RELEV_FML_UNUSED"
    input_parameters = (RankingModelsParameter, RelevFmlUnusedParameter,)

    @property
    def footer(self):
        if not self.is_completed():
            return None
        return [{
            "helperName": "",
            "content": """
<h4>Deprecated features that not used in any formula:<br/></h4>
<h4>%(features)s</h4>
            """ % {
                "features": self.ctx[UNUSED_DEPRECATED] if self.ctx[UNUSED_DEPRECATED] else "None",
            },
        }]

    @staticmethod
    def logging_result_file(result_file):
        logging.info("Stderr for relev_fml_unused tool:")
        for line in open(result_file, "r"):
            logging.info(line.strip("\n"))
        logging.info("End of logging relev_fml_unused stderr result")

    def get_unused_deprecated(self, result_file):
        unused_deprecated = []
        is_deprecated = False
        for line in open(result_file, "r"):
            line = line.strip("\n")
            if line.endswith(DEPRECATED_SUFFIX):
                is_deprecated = True
                continue
            if is_deprecated:
                items = line.split("\t")
                try:
                    assert(len(items) == 2)
                    int(items[0])
                    unused_deprecated.append(line)
                except:
                    break
        self.ctx[UNUSED_DEPRECATED] = "<br/>".join(unused_deprecated)

    def on_execute(self):
        build_task_name = "build_task"
        bundle_path = ""

        if build_task_name in self.ctx:
            bundleItem = rest.Client().resource.read(
                task_id=self.ctx[build_task_name],
                type=str(resource_types.ARCADIA_BINARY_ARCHIVE),
                state=resource.State.READY,
                limit=1,
            )["items"]
            if not len(bundleItem):
                raise Exception("No bundle in task #%s" % self.ctx[build_task_name])
            bundle_path = self.sync_resource(bundleItem[0]["id"])
        else:
            logging.info("Start build relev_fml_unused binary")
            subtask_for_build = self.create_subtask(
                task_type="BUILD_ARCADIA_BINARY",
                arch="linux",
                description="Build relev_fml_unused binary for RelevFmlUnused task #%s" % self.id,
                input_parameters={
                    "build_path": "tools/relev_fml_unused",
                    "binaries": "tools/relev_fml_unused/relev_fml_unused",
                },
            )
            self.ctx[build_task_name] = subtask_for_build.id
            self.wait_tasks(
                [subtask_for_build],
                list(self.Status.Group.FINISH + self.Status.Group.BREAK),
                wait_all=True
            )

        if not self.ctx[RankingModelsParameter.name]:
            self.ctx[RankingModelsParameter.name] = get_last_resource(resource_types.DYNAMIC_MODELS_ARCHIVE).id

        logging.info("Get models.archive from task #%s" % self.ctx[RankingModelsParameter.name])
        models_archive_path = self.sync_resource(self.ctx[RankingModelsParameter.name])

        file_with_result = "test_results"
        tmp_file = "tmp_file"

        # next line creates file for using as argument into relev_fml_unused option --formulas
        # this is legacy and should be deleted in new relev_fml_unused tool
        open(tmp_file, "a").close()
        more_options = " ".join(("-A", "./", "--formulas", tmp_file))

        commands = (
            ("tar -zxf %s" % bundle_path, "unpack"),
            ("./relev_fml_unused %(params)s -a %(models)s %(options)s 2> %(file)s || true" % {
                "params": self.ctx[RelevFmlUnusedParameter.name],
                "models": models_archive_path,
                "file": file_with_result,
                "options": more_options,
            }, "relev_fml_unused"),
        )

        for cmd, log_prefix in commands:
            logging.info("Start %s part of task with command %s" % (log_prefix, cmd))
            run_process(
                [cmd], work_dir="./", timeout=3600, shell=True, check=True,
                log_prefix=log_prefix, outputs_to_one_file=True,
            )

        self.logging_result_file(file_with_result)
        self.get_unused_deprecated(file_with_result)

        if len(self.ctx[UNUSED_DEPRECATED]):
            raise SandboxTaskFailureError("there are some TG_DEPRECATED features that not used in any formula, see footer for details")


__Task__ = RelevFmlUnused
