#!/usr/bin/env python

import logging
import os
import shutil

from sandbox.common import errors

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common import utils
from sandbox.projects.common.blender import settings


YA_PATH = os.path.join('.', 'ya')
RD_PATH = os.path.join('search', 'web', 'rearrs_upper', 'rearrange.dynamic')
PFC_PATH = os.path.join('yweb', 'blender', 'scripts', 'patch_fml_config', 'patch_fml_config.py')
REARRS_UPPER_TESTS_PATH = os.path.join('search', 'web', 'rearrs_upper', 'tests')
RM_FML_PATH = os.path.join(RD_PATH, 'rm_fml.py')

TESTS_PATHS = [REARRS_UPPER_TESTS_PATH]


class FormulaList(sp.SandboxStringParameter):
    name = "formula_list"
    description = "Whitespace-separated list of formulas to remove from rearrange.dynamic storage"
    default_value = ""
    multiline = True
    group = "Main parameters"


class DoPreModificationTests(sp.SandboxBoolParameter):
    name = "pre_tests"
    description = "Run arcadia tests before modifications"
    default_value = False
    group = "Main parameters"


class DoPostModificationTests(sp.SandboxBoolParameter):
    name = "post_tests"
    description = "Run arcadia tests after modifications"
    default_value = True
    group = "Main parameters"


class RunTEJobs(sp.SandboxBoolParameter):
    name = "run_te_jobs"
    description = "Run TestEnvironment tests from noapacheupper-trunk database on modified sources (not implemented yet)"
    default_value = False
    group = "Main parameters"


class WithSubformulas(sp.SandboxBoolParameter):
    name = "with_subformulas"
    description = "Remove also all unreferenced subformulas of the specified formulas"
    default_value = False
    group = "Removal parameters"


class DoCommit(sp.SandboxBoolParameter):
    name = "do_commit"
    description = "Do commit in the end"
    default_value = False
    group = "Commit parameters"


class LoginOverride(sp.SandboxStringParameter):
    name = "login_override"
    description = "Login to specify as the commit's diff-resolver. If left empty, task author will be used."
    default_value = ""
    group = "Commit parameters"


class CommitMessage(sp.SandboxStringParameter):
    name = "commit_message"
    description = "Commit message to use"
    default_value = "Remove rearrange.dynamic formulas"
    group = "Commit parameters"


class SshKeyOwner(sp.SandboxStringParameter):
    name = "ssh_key_owner"
    description = "Owner of the vault SSH key is in"
    default_value = settings.ROBOT_KEY_OWNER
    group = "Commit parameters"


class SshKeyName(sp.SandboxStringParameter):
    name = "ssh_key_name"
    description = "Name of the vault SSH key is in"
    default_value = settings.ROBOT_KEY_NAME
    group = "Commit parameters"


class SshUserName(sp.SandboxStringParameter):
    name = "ssh_user_name"
    description = "SSH user name to use for authorization"
    default_value = settings.ROBOT_NAME
    group = "Commit parameters"


class RemoveRearrangeDynamicFormulas(SandboxTask):
    type = "REMOVE_REARRANGE_DYNAMIC_FORMULAS"
    execution_space = 80000  # 80 Gb
    input_parameters = [
        FormulaList,
        DoPreModificationTests,
        DoPostModificationTests,
        RunTEJobs,
        WithSubformulas,
        DoCommit,
        LoginOverride,
        CommitMessage,
        SshKeyOwner,
        SshKeyName,
        SshUserName,
    ]

    def initial_checkout(self):
        Arcadia.checkout('arcadia:/arc/trunk/arcadia', self.working_copy, depth='immediates')

    def checkout_smth(self, path, log_prefix):
        process.run_process([YA_PATH, 'make', '--checkout', '-j0', path], work_dir=self.working_copy, check=True, log_prefix=log_prefix)

    def checkout_rd(self):
        self.checkout_smth(RD_PATH, "rearrange.dynamic checkout")

    def checkout_pfc(self):
        self.checkout_smth(os.path.dirname(PFC_PATH), "patch_fml_config checkout")

    def checkout_tests(self):
        for tests_path in TESTS_PATHS:
            self.checkout_smth(tests_path, "tests " + tests_path.replace(os.path.sep, '_') + " checkout")

    def build_smth(self, path, log_prefix):
        process.run_process([YA_PATH, 'make', '-r', path], work_dir=self.working_copy, check=True, log_prefix=log_prefix)

    def build_rd(self):
        self.build_smth(RD_PATH, "rearrange.dynamic build")

    def run_some_tests(self, path, log_prefix):
        process.run_process([YA_PATH, 'make', '-rA', '--checkout', '--test-disable-timeout', path], work_dir=self.working_copy, check=True, log_prefix=log_prefix)

    def run_tests(self):
        for tests_path in TESTS_PATHS:
            self.run_some_tests(tests_path, "tests " + tests_path.replace(os.path.sep, '_') + " run")

    def do_remove_fmls(self):
        fmls = utils.get_or_default(self.ctx, FormulaList).split()
        add_params = []
        if utils.get_or_default(self.ctx, WithSubformulas):
            add_params += ['--with-subformulas']
        process.run_process([RM_FML_PATH] + add_params + fmls, work_dir=self.working_copy, check=True, log_prefix="formulas removal")

    def get_diff(self):
        return Arcadia.diff(self.working_copy)

    def run_te_jobs(self):
        # TODO: Run noapacheupper-trunk TestEnvironment tests on modified sources
        logging.info("Not implemented yet")
        pass

    def do_commit(self, commit_message):
        with ssh.Key(self, utils.get_or_default(self.ctx, SshKeyOwner), utils.get_or_default(self.ctx, SshKeyName)):
            Arcadia.commit(self.working_copy, commit_message, utils.get_or_default(self.ctx, SshUserName))

    def on_execute(self):
        self.working_copy = 'arcadia'
        if os.path.isdir(self.working_copy):
            shutil.rmtree(self.working_copy)

        logging.info("Doing initial repo checkout")
        self.initial_checkout()

        logging.info("Doing checkout of used components")
        self.checkout_rd()
        self.checkout_pfc()
        if utils.get_or_default(self.ctx, DoPreModificationTests) or utils.get_or_default(self.ctx, DoPostModificationTests):
            self.checkout_tests()

        if utils.get_or_default(self.ctx, DoPreModificationTests):
            logging.info("Running pre-modification tests")
            self.run_tests()

        logging.info("Preparing working copy for formula removal")
        self.build_rd()

        logging.info("Doing formula removal")
        self.do_remove_fmls()

        res_diff = self.get_diff()
        logging.info("Resulting diff:\n" + res_diff)

        if utils.get_or_default(self.ctx, DoPostModificationTests):
            logging.info("Running post-modification tests")
            self.run_tests()

        if utils.get_or_default(self.ctx, RunTEJobs):
            logging.info("Running TE jobs on modified sources")
            self.run_te_jobs()

        diff_resolver = utils.get_or_default(self.ctx, LoginOverride).strip()
        if not diff_resolver:
            diff_resolver = self.author

        commit_message = "[diff-resolver:" + diff_resolver + "] " + utils.get_or_default(self.ctx, CommitMessage)
        if utils.get_or_default(self.ctx, DoPostModificationTests):
            commit_message += " SKIP_CHECK"
        logging.info("Commit message: " + commit_message)

        if utils.get_or_default(self.ctx, DoCommit):
            logging.info("Commititng as " + diff_resolver)
            try:
                self.do_commit(commit_message)
            except Exception as e:
                raise errors.TemporaryError(e)
        else:
            logging.info("Not committing as " + diff_resolver)


__Task__ = RemoveRearrangeDynamicFormulas
