#  -*- coding: utf-8 -*-

from __future__ import absolute_import
from sandbox import common
import logging
import json
import re
import datetime
import time

from sandbox.projects import resource_types
import sandbox.projects.common.utils
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter, SandboxSelectParameter, ResourceSelector
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.channel import channel


class Revision(SandboxIntegerParameter):
    description = 'From HEAD to revision:'
    name = 'revision'
    default_value = 2559767


class Resource(ResourceSelector):
    name = 'last_run'
    resource_type = resource_types.REPORT_CHECK_AUTOTEST_STATUS
    description = 'Resource with revision:'


class Selector(SandboxSelectParameter):
    name = 'selector'
    choices = [('manualy', 'manualy'), ('from resource', 'from_resource'), ('auto', 'auto')]
    sub_fields = {'manualy': [Revision.name], 'from_resource': [Resource.name]}
    default_value = choices[0][1]
    description = 'How to define revision for stop:'


class Email(SandboxBoolParameter):
    name = 'send_email'
    description = 'Send email to the authors of commits'
    default_value = False
    do_not_copy = True


class ReportCheckAutotestStatus(SandboxTask, object):
    """
        Проверяет что в тасках репорта сделали автотесты.
    """

    type = 'REPORT_CHECK_AUTOTEST_STATUS'

    input_parameters = [Selector, Revision, Resource, Email]

    def on_execute(self):
        select = self.ctx[Selector.name]

        if select == 'manualy':
            serp_list = self.manualy(self.ctx[Revision.name])
        elif select == 'from_resource':
            serp_list = self.from_resource(self.ctx[Resource.name])
        elif select == 'auto':
            serp_list = self.auto()
        else:
            raise SandboxTaskFailureError("Unknowk how to process param %s" % select)

        stop_rev = serp_list["last_revision"]
        url = Arcadia.normalize_url('svn+ssh://arcadia/arc/trunk/arcadia/web/report')
        log_list = Arcadia.log(url, 'HEAD', stop_rev)
        # [
        #   {
        #     'date': datetime.datetime(2016, 11, 15, 15, 49, 52, 23079),
        #     'msg': 'SERP-47959: Added UNIT tests',
        #     'paths': [('M', '/trunk/arcadia/web/report/t/YxWeb/Util/R13n/1.t')],
        #     'author': 'ilvin',
        #     'revision': 2560050
        #     }
        # ]

        now = int(time.time())
        now2 = datetime.datetime.utcfromtimestamp(now)
        tomorrow = datetime.datetime(now2.year, now2.month, now2.day, 12, 0, 0)+datetime.timedelta(days=1)
        tomorrow = int(time.mktime(tomorrow.utctimetuple()))

        # если хотим обрабатывать не только очередь SERP, то нужно написать заявку
        # на tools@ чтобы выдали доступ роботу zomb-prj-10 до нужной очереди
        p = re.compile(r"\bSERP-\d+\b")
        if log_list:
            serp_list["last_revision"] = int(log_list[0]['revision'])
            for item in log_list:
                author = item["author"]
                # может быть несколько задач в коммите
                for name in p.findall(item["msg"]):
                    name = str(name)
                    if name not in serp_list["task"]:
                        serp_list["task"][name] = {}

                    send_time = serp_list["task"][name].get(author, tomorrow)
                    serp_list["task"][name][author] = send_time

        st = common.rest.Client(
            base_url='https://st-api.yandex-team.ru/v2',
            auth=self.get_vault_data("REPORT_CORE", "zomb-prj-10-st-oauth")
        )

        message_body = []
        for name in serp_list["task"].keys():
            try:
                st_task = st.issues[name].read()
            except st.HTTPError as e:
                message_body.append('%s: %s' % (name, e.response.text))
                continue

            if st_task:
                auto = st_task.get("autotesting", "").encode("utf-8")
                status = st_task["status"]["key"]
                if status == "closed" or auto == 'Тест написан' or auto == 'Тестирование не требуется':
                    serp_list["task"].pop(name)

        # send email to serp-core-tests@
        if message_body:
            now = datetime.datetime.utcfromtimestamp(int(time.time()))
            mail_to = 'serp-core-test@yandex-team.ru'
            mail_subject = 'Some problem with %s task' % self.type
            # https://st.yandex-team.ru/SERP/description
            mail_body = 'Sandbox task https://sandbox.yandex-team.ru/task/%s/view has problem.\nPlease check st tasks:\n' % self.id + '\n'.join(message_body)
            logging.error('mail-to: %s, mail_subject: %s, body: %s' % (mail_to, mail_subject, mail_body))
            # now.isoweekday() Monday == 1 ... Sunday == 7
            if self.scheduler and now.hour >= 10 and now.hour <= 13 and now.isoweekday() < 6:
                channel.sandbox.send_email(mail_to, None, mail_subject, mail_body, content_type='text/plain', charset='utf-8')
            raise SandboxTaskFailureError('Can not check some task in the ST')
        # --------

        # обновить время, подготовить список для рассылки
        email = {}
        for name in serp_list["task"]:
            for author in serp_list["task"][name]:
                if serp_list["task"][name][author] <= now:
                    serp_list["task"][name][author] = tomorrow
                    if not email.get(author):
                        email[author] = []
                    email[author].append(name)

        logging.info("email=%s data=%s" % (email, serp_list))
        res_path = 'serp_without_tests.json'
        with open(res_path, "w") as f:
            f.write(json.dumps(serp_list))

        attrs = {}
        if self.scheduler:
            logging.info(self.scheduler)
            attrs["scheduler"] = self.scheduler
        # создаем ресурс
        self.create_resource(
            description=self.descr,
            resource_path=res_path,
            resource_type=resource_types.REPORT_CHECK_AUTOTEST_STATUS,
            attributes=attrs
        )

        if self.ctx[Email.name] and email:
            for author in email:
                subject = 'SERP-XXXX fill in the field Autotesting'
                body = '<html><body><pre>Dear %s\nPlease change the state of field autotesting in tasks:\n' % author
                for name in email[author]:
                    # summary
                    body += '<a href="https://st.yandex-team.ru/%s">%s</a>\n' % (name, name)
                body += '</pre></body></html>'

                channel.sandbox.send_email(
                    mail_to=author + '@yandex-team.ru',
                    mail_cc=None,
                    mail_subject=subject,
                    mail_body=body,
                    content_type='text/html',
                )

    def manualy(self, revision):
        serp_list = {"task": {}, "last_revision": revision}
        return serp_list

    def from_resource(self, res_id):
        sandbox.projects.common.utils.sync_resource(res_id)
        resource = channel.sandbox.get_resource(res_id)
        if not resource:
            raise SandboxTaskFailureError('Cannot execute task. Can\'t find resource by id %s.' % res_id)

        with open(resource.path, "r") as f:
            serp_list = json.loads(f.read())

        return serp_list

    def auto(self):
        # найти последний успешный таск, запущенный планировщиком
        # если нашли, то взять ресурс и использовать его, если нет то используем ревизию по умолчанию
        if not self.scheduler:
            raise SandboxTaskFailureError('Task MUST be run only by scheduler')

        param = {
            "scheduler": self.scheduler,
            "status": ctt.Status.SUCCESS,
            "limit": 1,
            "order": "-id"
        }
        res = common.rest.Client().task.read(param)
        if not res or not res["items"]:
            logging.info("use default revision")
            return self.manualy(Revision.default_value)

        param = {
            "task_id": res["items"][0]["id"],
            "type": resource_types.REPORT_CHECK_AUTOTEST_STATUS.name,
            "state": ctr.State.READY,
            "limit": 1,
            "order": "-id"
        }

        res = common.rest.Client().resource.read(param)
        if not res or not res["items"]:
            raise SandboxTaskFailureError('Can not find resource with type %s in the task: %s' % (param["type"], param["task_id"]))

        logging.info("use resource %s" % res["items"][0]["id"])
        return self.from_resource(res["items"][0]["id"])


__Task__ = ReportCheckAutotestStatus
