#  -*- coding: utf-8 -*-

import os
import logging
import json
import time
import copy
import threading

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxSubprocessError
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxStringParameter, LastReleasedResource
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxRadioParameter
from sandbox.sandboxsdk.parameters import Container
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.channel import channel
import sandbox.projects.TestReportUnit as Unit
from sandbox.projects.common import utils
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.report.common import ApacheBundleParameter, Project
import sandbox.projects.sandbox
from sandbox.projects.common import apihelpers


class SourceException(Exception):
    def __init__(self, exc):
        self.__exc = exc

    def get_exception(self):
        return self.__exc


class ProdSelector(SandboxRadioParameter):
    required = True
    name = 'data_for_production'
    description = 'Generate data for:'

    choices = [
        # name value
        ('beta', 'beta'),
        ('production', 'production'),
    ]
    per_line = 2
    default_value = choices[0][1]


class Module(SandboxStringParameter):
    required = True
    name = 'module'
    description = 'Generate data for module:'
    default_value = 'YxWeb::Util::LCookieKeys'


class CacheSource(LastReleasedResource):
    required = False
    name = 'source_id'
    description = 'Cache source. Generate data using this source:'
    resource_type = resource_types.REPORT_DATA_RUNTIME_ITEM_SOURCE


class DependsOnResource(LastReleasedResource):
    name = 'depends_on_resources'
    description = "Depends on resources:"
    required = False
    resource_type = [resource_types.REPORT_DATA_RUNTIME_ITEM]


class ReportCoreParameter(Unit.DefaultUpperSearchParams.ReportCoreParameter):
    resource_type = [
        resource_types.REPORT_CORE_PACKAGE,
        resource_types.REPORT_IMAGES_CORE_PACKAGE,
        resource_types.REPORT_NEWS_CORE_PACKAGE,
        resource_types.REPORT_VIDEO_CORE_PACKAGE,
        resource_types.REPORT_YACA_CORE_PACKAGE,
    ]


class Selector(Unit.Selector):
    choices = [('svn', 'svn'), ('package', 'package')]
    sub_fields = {'svn': [Unit.ArcadiaUrl.name], 'package': [ReportCoreParameter.name]}
    default_value = choices[0][1]
    description = 'Get report from:'
    required = True


class UseExpiredSource(SandboxBoolParameter):
    name = 'use_expired_sourse'
    description = 'Use expired source if update source failed'
    default_value = False
    do_not_copy = True


class IsExpired(object):
    name = 'is_expired'


class ContainerLxc(Container):
    description = 'lxc container:'

    @property
    def default_value(self):
        res = apihelpers.get_last_resource_with_attribute(
            sandbox.projects.sandbox.LXC_CONTAINER,
            attribute_name='report',
            attribute_value='1',
        )
        return res.id if res else None


class ReportDataRuntimeItem(SandboxTask, object):
    """
       Для указанного модуля создает ресурсы с источниками и данными(data.runtime).  Вспомагательный таск.
    """

    type = 'REPORT_DATA_RUNTIME_ITEM'

    environment = (
        environments.SvnEnvironment(),
    )

    cores = 1
    required_ram = 8072
    execution_space = 7000
    report_dir = 'report'
    apache_bundle = 'apache_bundle'
    tar_zip = 0

    input_parameters = [
        ProdSelector, Project, Module, ApacheBundleParameter, Selector, Unit.ArcadiaUrl, ReportCoreParameter,
        CacheSource, UseExpiredSource, DependsOnResource, Unit.RtccBundle, ContainerLxc
    ]

    @staticmethod
    def change_resource_description(resource_id, new_description):
        try:
            channel.rest.server.resource[resource_id] = {"description": new_description}
        except channel.rest.server.HTTPError as exc:
            raise SandboxTaskFailureError(
                'Cannot change description for resource {}: {}'.format(resource_id, exc)
            )
        return resource_id

    @property
    def available_modules(self):
        if not getattr(self, '_available_modules', None):
            self._available_modules = self.modules('all')
        return self._available_modules

    @property
    def versions(self):
        if not getattr(self, '_versions', None):
            self._versions = self.find_versions()
        return self._versions

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        if 'data_resource_id' not in self.ctx:
            channel.task = self
            self.ctx['data_resource_id'] = self.create_resource(
                resource_type=resource_types.REPORT_DATA_RUNTIME_ITEM,
                resource_path='data.tar.gz',
                description='test',
            ).id

    def on_execute(self):
        cached_source = self.check_input()
        self.my_prepare()
        self.main(cached_source)

    def check_input(self):
        if not self.ctx[Selector.name]:
            raise SandboxTaskFailureError("Field '%s' can not be empty" % Selector.description)

        # если репорт из пакета то тип ресурса должен соответствовать проекту
        if self.ctx[Selector.name] == 'package':
            res_id = self.ctx[ReportCoreParameter.name]
            resource = channel.rest.get_resource(res_id)
            if not resource:
                raise SandboxTaskFailureError('Can not find resource with id=%s' % res_id)

            report_type = Project.project_to_report[self.ctx[Project.name]]
            if resource.type != report_type:
                raise SandboxTaskFailureError('Wrong report type. For project %s need report %s' % (self.ctx[Project.name], report_type))

        # ctx[Module.name] должен быть только один модуль
        if len(self.ctx[Module.name].split()) > 1:
            raise SandboxTaskFailureError("Support only one module")

        # в ресурсе CacheSource атрибут module должен совпадать со значением контекста задачи
        # источники не зависят от project, configuration поэтому нужно проверить только module. Источники характеризуются атрибутом md5_uri
        resource = None
        if self.ctx[CacheSource.name]:
            res_id = self.ctx[CacheSource.name]
            resource = channel.rest.get_resource(res_id)
            if not resource:
                raise SandboxTaskFailureError('Can not find resource with id=%s' % res_id)

            for attr_name, ctx_name in [['module', Module.name]]:
                attr_value = resource.attributes.get(attr_name)
                if not attr_value or self.ctx[ctx_name] != attr_value:
                    raise SandboxTaskFailureError("Resource %s has wrong attribute %s=%s. Need %s=%s" % (resource, attr_name, attr_value, attr_name, self.ctx[ctx_name]))

        if self.ctx[UseExpiredSource.name] and not self.ctx[CacheSource.name]:
            raise SandboxTaskFailureError("Can not find cache source")

        return resource

    def my_prepare(self):
        self.set_env()

        # фиксируем ревизию
        Unit.TestReportUnit.freeze_svn_url(self)
        # развернуть в рабочей директории задачи - репорт и upper
        if not self.ctx[ApacheBundleParameter.name]:
            self.ctx[ApacheBundleParameter.name] = Unit.TestReportUnit._get_last_released_resource('APACHE_BUNDLE').id
            logging.info('Using APACHE_BUNDLE: %s', self.ctx[ApacheBundleParameter.name])
        Unit.TestReportUnit.get_apache(self.ctx[ApacheBundleParameter.name], self.apache_bundle)
        Unit.TestReportUnit.get_report(self, self.report_dir)
        # получить upper_conf
        Unit.TestReportUnit.get_upper_config(self.ctx[Unit.RtccBundle.name], project=self.ctx[Project.name], is_beta=(self.ctx[ProdSelector.name] == 'beta'))

        if self.ctx[Selector.name] != 'svn':
            # если репорт из пакета то нет возможности выполнить svn info
            # так же в контексте таска может не быть информации о ревизии(есть таски которые копируют ресурсы)
            prog = "y-local-env perl -e 'use YxWeb::Setup;use JSON; $YxWeb::FULL_VERSION_REPORT =~ m!^URL:\s+(.+)\s*$!m; die q{Can not find svn URL} unless $1; my $out = {url => $1};\
$YxWeb::FULL_VERSION_REPORT =~ m!^Revision: (\d+)$!m; die q{Can not find svn revision} unless $1; $out->{revision} = $1; print JSON->new->encode($out)'"
            data = json.loads(Unit.TestReportUnit.run_and_read(prog, self.report_dir))
            logging.debug("report svn url and revision from YxWeb::FULL_VERSION_REPORT: %s" % data)
            self.ctx[Unit.ArcadiaUrlFrozen.name] = Arcadia.normalize_url("{url}@{revision}".format(**data))

    def set_env(self):
        env_vars = {}
        if self.ctx[ProdSelector.name] == 'beta':
            env_vars.update({
                'IS_BETA': '1',
                'YX_IS_BETA': '1',
            })

        env_vars.update({'YX_PROJECT': self.ctx[Project.name]})

        for varname, varval in env_vars.iteritems():
            os.environ[varname] = varval
            logging.debug('{} => {}'.format(varname, os.getenv(varname)))

    def main(self, cached_source):
        # узнать где data.runtime, data.permanent, data.runtime/.source - все пути относительные
        logging.info("data_path=%s" % self.find_source_path)

        # если есть зависимые ресурсы скопировать их
        if self.ctx[DependsOnResource.name]:
            make_folder(self.find_source_path['runtime'])

            def thread_copy(res_id, res_path, error):
                try:
                    src_path = self.sync_resource(res_id)
                    res_path[res_id] = src_path
                except Exception as e:
                    error.append("Can not sync res_id: %s. Exception: %s" % (res_id, e))
                    raise e

            pool_error = []
            res_path = {}
            for res_id in self.ctx[DependsOnResource.name]:
                res_path[res_id] = ""

            # копируем ресурсы параллельно
            pool = [threading.Thread(target=thread_copy, args=(res_id, res_path, pool_error)) for res_id in self.ctx[DependsOnResource.name]]
            map(threading.Thread.start, pool)
            map(threading.Thread.join, pool)
            if pool_error:
                raise SandboxTaskFailureError("%s" % pool_error)

            # распаковываем последовательно
            for res_id in res_path:
                self.unpack_data(res_path[res_id])

        run_process(['y-local-env perl scripts/make.pl --mode=setup --generator --verbose all'], shell=True, log_prefix='setup', work_dir=self.report_dir)
        # узнать список модулей
        modules = self.modules(self.ctx[Module.name])
        if not len(modules):
            raise SandboxTaskFailureError("Can't find module %s" % self.ctx[Module.name])
        elif len(modules) > 1:
            raise SandboxTaskFailureError("Support only one module")

        for mod_name in modules:
            source = None
            # создаем источник только если нет кеша или выставлена опция UseExpiredSource
            if not self.ctx[CacheSource.name] or self.ctx[UseExpiredSource.name]:
                logging.info("Try generate source")
                try:
                    source = self.create_source(mod_name)
                except SourceException as e:
                    if not self.ctx[UseExpiredSource.name]:
                        raise e.get_exception()

            if not source:
                if self.parent_id and self.ctx[UseExpiredSource.name]:
                    self.ctx[IsExpired.name] = 1
                    self.descr += " (Use expired source)"

                logging.info("Get source for module %s from cache" % mod_name)
                source = cached_source

            self.create_data_from_source(mod_name, source)

    def modules(self, mod_name):
        modules = json.loads(Unit.TestReportUnit.run_and_read('y-local-env perl scripts/make.pl --mode=extended_alias --verbose %s' % mod_name, self.report_dir))
        # convert list into tuple
        for mod_name in modules:
            if modules[mod_name]['depends']:
                modules[mod_name]['depends'] = tuple(modules[mod_name]['depends'])
        logging.debug("modules=%s" % modules)

        return modules

    def find_versions(self):
        prog = 'y-local-env perl scripts/make.pl --version-full'
        data = Unit.TestReportUnit.run_and_read(prog, self.report_dir)
        logging.debug("{} =>\n{}" .format(prog, data))

        # TODO оптимизировать вызовы
        js_data = json.loads(Unit.TestReportUnit.run_and_read('y-local-env perl scripts/make.pl --environment', self.report_dir))
        v = {
            'runtime_version': Unit.TestReportUnit.runtime_version_for_report(self.report_dir),
            'report_version': js_data['version_report'] if 'version_report' in js_data else 'unknown',
        }
        # r77.4 r77.HEAD trunk
        # v = {'runtime_version': 'trunk'}
        logging.info("versions=%s" % v)
        return v

    def get_relpath(self, path, start=None):
        if start is None:
            start = self.abs_path()

        return os.path.relpath(os.path.realpath(path), os.path.realpath(start))

    @property
    def find_source_path(self):
        if not getattr(self, '_find_source_path', None):
            # узнать где data.runtime, data.permanent, data.runtime/.source, data.runtime/.metainfo
            prog = "y-local-env perl -MYxWeb::Setup -e '\
use JSON;\
use YxWeb::Util::Data::MetaInfo;\
print JSON->new->encode([$YxWeb::RUNTIME_DATA_DIR, $YxWeb::PERMANENT_DATA_DIR, $YxWeb::RUNTIME_SOURCE_DIR, YxWeb::Util::Data::MetaInfo::DIR])'"
            # data_runtime_path, data_permanent_path, data_source_path = json.loads(Unit.TestReportUnit.run_and_read(prog, self.report_dir))
            data = json.loads(Unit.TestReportUnit.run_and_read(prog, self.report_dir))
            # получить относительные пути
            for i in range(len(data)):
                data[i] = self.get_relpath(data[i])

            self._find_source_path = {
                "runtime":   data[0],
                "permanent": data[1],
                "source":    data[2],
                "metainfo":  data[3],
            }

        return self._find_source_path

    def create_source(self, mod_name):
        mod_path = self.available_modules[mod_name]['path']
        ctx_name = "source/%s" % mod_path
        if self.ctx.get(ctx_name):
            logging.info("Resource with source for module %s already created" % mod_name)
            return channel.rest.get_resource(self.ctx[ctx_name])
        else:
            # не у всех модулей есть источники, поэтому создадим директорию, чтобы создать пустой ресурс
            data_folder = os.path.join(self.find_source_path['source'], mod_path)
            make_folder(data_folder, delete_content=True)

            # resource_dir директория куда будем складывать ресурсы
            resource_dir = 'resource'
            resource_path = os.path.join(resource_dir, mod_path)
            make_folder(resource_path, delete_content=True)
            resource_path = os.path.join(resource_path, 'source-%s.tar' % self.versions['runtime_version'])

            try:
                run_process(['y-local-env perl scripts/make.pl --mode=make --generator --sources --verbose --sandbox %s' % (mod_name,)], shell=True, log_prefix='make', work_dir=self.report_dir)
            except SandboxSubprocessError as e:
                raise SourceException(e)

            tar_params = "-cf"
            if self.tar_zip:
                tar_params = "-czf"
                resource_path += '.gz'

            run_process(['tar %s %s %s' % (tar_params, resource_path, data_folder)], shell=True, log_prefix='tar')

            # узнать хеш от uri источников
            md5_uri = self.available_modules[mod_name]["md5_uri"]
            build_time = int(time.time())
            attrs = {
                'version': self.versions['runtime_version'],
                'project': self.ctx[Project.name],
                'configuration': self.ctx[ProdSelector.name],
                'module': mod_name,
                'md5_uri': md5_uri,
                'build_time': build_time,
                'expire_time': self.expire_time({mod_name: build_time})[mod_name]["next"],
            }
            # для создания ресурса нужен относительный путь
            resource = self.create_resource(
                resource_type=resource_types.REPORT_DATA_RUNTIME_ITEM_SOURCE,
                resource_path=resource_path,
                description=mod_name,
                attributes=attrs
            )
            self.mark_resource_ready(resource.id)
            self.ctx[ctx_name] = resource.id
            logging.info("Resource %s for module %s was created successfully" % (resource.type, mod_name))
            return resource

    def expire_time(self, modules):
        path_to_name = {}
        echo_str = []
        for mod_name in modules:
            mod_path = self.available_modules[mod_name]['path']
            path_to_name[mod_path] = mod_name
            build_time = modules[mod_name]
            echo_str.append("%s=%s" % (mod_path, build_time))

        data = json.loads(Unit.TestReportUnit.run_and_read('y-local-env perl scripts/make.pl --environment', self.report_dir))

        filename = [os.path.join(self.abs_path(), "util_random_text.cron")]
        f = open(filename[0], "w")
        f.write("0 5 * * * util/random_text")
        f.close()

        for name in ("cron_generator", ):
            val = data.get(name)
            if val:
                filename.append(val)

        prog = 'echo "%s" | y-local-env perl scripts/dev/cron.pl %s' % ('\\n'.join(echo_str), ' '.join(filename))
        data = json.loads(Unit.TestReportUnit.run_and_read(prog, self.report_dir))
        # {"wizard/pseudo/fast": {"next":1433320800, "expired":1} }

        expire_time = {}
        for mod_path in data:
            mod_name = path_to_name[mod_path]
            expire_time[mod_name] = data[mod_path]

        logging.debug(expire_time)
        return expire_time

    def module_path(self, mod_name):
        mod_path = self.available_modules[mod_name]['path']
        # data_folder
        return os.path.join(self.find_source_path['runtime'], mod_path)

    def write_module_metainfo(self, mod_name, info):
        f = open(os.path.join(self.module_path(mod_name), '.sandbox_metainfo.json'), "w")
        f.write(json.dumps(info, sort_keys=True, indent=4))
        f.close()
        logging.info("Metainfo for module %s was created" % mod_name)

    def module_metainfo(self, mod_name):
        f = open(os.path.join(self.module_path(mod_name), '.sandbox_metainfo.json'))
        data = json.loads(f.read())
        f.close()
        return data

    def create_data_from_source(self, mod_name, source):
        mod_path = self.available_modules[mod_name]['path']
        # копируем ресурс
        src_path = self.sync_resource(source.id)
        self.unpack_source(src_path)

        ctx_name = "data/%s" % mod_path
        if self.ctx.get(ctx_name):
            logging.info("Resource with data for module %s already created" % mod_name)
        else:
            run_process(['y-local-env perl scripts/make.pl --mode=make --generator --cached --verbose --sandbox %s' % (mod_name,)], shell=True, log_prefix='make', work_dir=self.report_dir)
            run_process(['y-local-env perl scripts/make.pl --mode=check --generator --verbose %s' % (mod_name)], shell=True, log_prefix='make', work_dir=self.report_dir)

            attrs = {
                'version': self.versions['runtime_version'],
                'revision': self.modules_revision([mod_name])[mod_name],
                'project': self.ctx[Project.name],
                'configuration': self.ctx[ProdSelector.name],
                'module': mod_name,
                'source_id': str(source.id),
                'build_time': int(time.time()),
                'expire_time': int(source.attributes['expire_time']),
                'path': self.available_modules[mod_name]['path'],
            }
            if self.ctx[DependsOnResource.name]:
                attrs['depend_on'] = ','.join([str(i) for i in self.ctx[DependsOnResource.name]])

            for attr_name in ('version', 'build_time', 'md5_uri'):
                attrs["source_" + attr_name] = source.attributes[attr_name]

            # собрать метаинфо
            meta_info = copy.copy(attrs)
            for attr_name in ("build_time", "source_build_time", "expire_time"):
                meta_info[attr_name] = int(meta_info[attr_name])
                meta_info[attr_name + "_utc"] = time.strftime("%Y-%m-%d %H:%M:%S UTC", time.gmtime(meta_info[attr_name]))

            self.write_module_metainfo(mod_name, meta_info)

            # resource_dir директория куда будем складывать ресурсы
            resource_dir = 'resource'
            resource_path = os.path.join(resource_dir, mod_path)
            make_folder(resource_path)
            resource_path = os.path.join(resource_path, 'data-%s.tar' % self.versions['runtime_version'])
            tar_params = "-cf"
            if self.tar_zip:
                tar_params = "-czf"
                resource_path += '.gz'

            data_folder = self.module_path(mod_name)
            run_process(['tar %s %s %s' % (tar_params, resource_path, data_folder)], shell=True, log_prefix='tar')

            resource = channel.rest.get_resource(self.ctx['data_resource_id'])
            self.change_resource_basename(resource.id, resource_path)
            self.change_resource_description(resource.id, mod_name)
            utils.set_resource_attributes(resource, attrs)

            self.ctx[ctx_name] = resource.id
            logging.info("Resource %s for module %s was created successfully" % (resource.type, mod_name))

    def unpack_source(self, src_path):
        tar_params = "-xf"
        if os.path.basename(src_path).endswith('.gz'):
            tar_params = "-xzf"
        # положить ресурс с исходником в self.find_source_path['source']
        run_process(['tar -C %s --strip-components=2 %s %s' % (self.find_source_path['source'], tar_params, src_path)], shell=True, log_prefix='tar')

    def unpack_data(self, src_path):
        tar_params = "-xf"
        if os.path.basename(src_path).endswith('.gz'):
            tar_params = "-xzf"
        # положить ресурс с исходником в self.find_source_path['runtime']
        run_process(['tar -C %s --strip-components=1 %s %s' % (self.find_source_path['runtime'], tar_params, src_path)], shell=True, log_prefix='tar')

    def modules_revision(self, modules):
        # для модуля определить от каких файлов (перловых модулей) он зависит. К сожалению часть модулей загружается на этапе
        # выполнения(эти модули перечислены в self.dependence), из=за этого в %INC не все зависимости. Поэтому нужно руками добавить
        # необходимые use(рекурсивно обходить не надо!)
        # для каждого файла узнаем COMMITTED ревизию и берем максимальную, эту максимальную ревизию и будем использовать
        # в качестве версии для данных
        new = []
        if not getattr(self, '_modules_revision', None):
            self._modules_revision = {}

        for mod_name in modules:
            if mod_name not in self.available_modules:
                raise SandboxTaskFailureError("Module %s not in available_modules" % mod_name)

            if mod_name not in self._modules_revision:
                new.append(mod_name)

        if new:
            prog = 'y-local-env perl scripts/make.pl --verbose --mode module_revision %s' % ' '.join(new)
            if self.ctx[Selector.name] != 'svn':
                prog += " --svn_report_url %s" % Arcadia.svn_url(self.ctx[Unit.ArcadiaUrlFrozen.name])
            rev = json.loads(Unit.TestReportUnit.run_and_read(prog, self.report_dir))
            self._modules_revision.update(rev)
        return self._modules_revision

    @property
    def dependence(self):
        if not getattr(self, '_dependence', None):
            # для создания даных для модуля 'YxWeb::Util::Region' нужны данные модулей 'YxWeb::Util::Category'
            # self._dependence = {
            #     раз в сутки              раз в сутки
            #     'YxWeb::Util::Region': ('YxWeb::Util::Category',),
            #     раз в сутки               раз в сутки              раз в сутки             раз в сутки            раз в сутки
            #          раз в сутки                      раз в сутки
            #     'YxWeb::Wizard::Adresa': ('YxWeb::Util::Category', 'YxWeb::Util::GeoBase', 'YxWeb::Util::Region', 'YxWeb::Util::DeviceDetect',\
            #          'YxWeb::Util::TrustedNet::User', 'YxWeb::Util::TrustedNet::Server'),
            #     раз в 30 мин          раз в сутки            раз в сутки
            #     'YxWeb::Data::News': ('YxWeb::Util::Region', 'YxWeb::Util::GeoBase'),
            #     раз в 5 мин                  раз в сутки            раз в сутки
            #     'YxWeb::Util::Experiments': ('YxWeb::Util::Region', 'YxWeb::Util::GeoBase'),
            #     раз в сутки                           раз в сутки            раз в сутки             раз в сутки              раз в 30 мин                      раз в сутки
            #     'YxWeb::Util::ReqTokens::Localized': ('YxWeb::Util::Region', 'YxWeb::Util::GeoBase', 'YxWeb::Util::Category', 'YxWeb::Wizard::Afisha::Rubrics', 'YxWeb::Wizard::Adresa'),
            #     раз в сутки            раз в сутки              раз в сутки             раз в сутки
            #     'YxWeb::Report::Web': ('YxWeb::Util::Category', 'YxWeb::Util::GeoBase', 'YxWeb::Util::Region')
            # }

            self._dependence = {}
            for mod_name in self.available_modules:
                if self.available_modules[mod_name]['depends']:
                    self._dependence[mod_name] = self.available_modules[mod_name]['depends']

            # проверить список на циклические зависимости
            for name in self._dependence:
                self.check_cycle(name)

        return self._dependence

    def check_cycle(self, name, dependent=None):
        if dependent is None:
            dependent = []

        if name in dependent:
            dependent.append(name)
            raise SandboxTaskFailureError("Module %s. Cycle dependence %s." % (name, dependent))

        if name in self._dependence:
            for dep_name in self._dependence[name]:
                newdependent = copy.copy(dependent)
                newdependent.append(name)

                self.check_cycle(dep_name, newdependent)


__Task__ = ReportDataRuntimeItem
