#  -*- coding: utf-8 -*-

import os
import re
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter, SandboxBoolParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common import apihelpers

import sandbox.projects.ReportDataRuntimeItem as RDRI
import sandbox.projects.ReportDataRuntime as RDR
import sandbox.projects.TestReportUnit as Unit
import logging
from sandbox import common
from sandbox.projects import resource_types
import threading

from sandbox.projects.common.nanny import nanny
import sandbox.projects.report.common as report_common

# ('WEB', 'NEWS', 'IMGS', 'VIDEO', 'YACA')
project_list = RDRI.Project.available_projects
modes = ('branch', 'tag')
regex = {
    modes[0]: {
        project_list[0]: {
            'regexp': re.compile('^r(\d+)/$'),
            'path': 'report',
            'regexp_minor': re.compile('^(\d+)(?:\.(\d+))?/$'),
        },
        project_list[1]: {
            'regexp': re.compile('^r(\d+)_news/$'),
            'path': 'report',
            'regexp_minor': re.compile('^(\d+)(?:\.(\d+))?/$'),
        },
        project_list[2]: {
            'regexp': re.compile('^r(\d+)/$'),
            'path': 'report/images',
            'regexp_minor': re.compile('^(\d+)(?:\.(\d+))?/$'),
        },
        project_list[3]: {
            'regexp': re.compile('^r(\d+)/$'),
            'path': 'report/video',
            'regexp_minor': re.compile('^(\d+)(?:\.(\d+))?/$'),
        },
        project_list[4]: {
            'regexp': re.compile('^r(\d+)/$'),
            'path': 'report/yaca',
            'regexp_minor': re.compile('^(\d+)(?:\.(\d+))?/$'),
        },
    },
    modes[1]: {
        project_list[0]: {
            'regexp': re.compile('^r(\d+)/$'),
            'path': 'report',
            'regexp_minor': re.compile('^(\d+)(?:\.(\d+))?/$'),
        },
        project_list[1]: {
            'regexp': re.compile('^r(\d+)_news/$'),
            'path': 'report',
            'regexp_minor': re.compile('^(\d+)(?:\.(\d+))?/$'),
        },
        project_list[2]: {
            'regexp': re.compile('^r(\d+)_images/$'),
            'path': 'report',
            'regexp_minor': re.compile('^(\d+)(?:\.(\d+))?/$'),
        },
        project_list[3]: {
            'regexp': re.compile('^r(\d+)_video/$'),
            'path': 'report',
            'regexp_minor': re.compile('^(\d+)(?:\.(\d+))?/$'),
        },
        project_list[4]: {
            'regexp': re.compile('^r(\d+)_yaca/$'),
            'path': 'report',
            'regexp_minor': re.compile('^(\d+)(?:\.(\d+))?/$'),
        },
    },
}


class NumberTags(SandboxIntegerParameter):
    name = 'numder_tags'
    description = 'Generate data for last N tags(0 - not generate):'
    default_value = 3
    required = True


class NumberBranches(SandboxIntegerParameter):
    name = 'numder_branch'
    description = 'Generate data for last K branches(0 - not generate):'
    default_value = 3
    required = True


class NumberMinorTags(SandboxIntegerParameter):
    name = 'numder_minor_tags'
    description = 'Generate data for last M minor tags(0 - unlim):'
    default_value = 10
    required = True


class CreateBundle(SandboxBoolParameter):
    name = 'create_bundle'
    description = 'Create bundle'
    default_value = False


class ApacheBundleParameter(report_common.ApacheBundleParameter):
    required = False
    default_value = None


class ReportDataRuntimeTags(nanny.ReleaseToNannyTask, SandboxTask, object):
    """
       Создает данные(data.runtime) для:
         1) последних N мажорных тегов(для M минорных) (configuration=production)
         2) последних K бранчей (configuration=beta)
       Опционально собирает бандл из ресурсов созданных в п1
    """

    type = 'REPORT_DATA_RUNTIME_TAGS'
    execution_space = 3000
    cores = 1
    required_ram = 3000

    input_parameters = [ApacheBundleParameter, RDRI.Project, NumberTags, NumberMinorTags, NumberBranches, CreateBundle]

    DATA_RESOURCES_TYPE_BY_PROJECT = {
        'WEB': resource_types.WEB_REPORT_DATA_RUNTIME_BUNDLE,
        'IMGS': resource_types.IMGS_REPORT_DATA_RUNTIME_BUNDLE,
        'VIDEO': resource_types.VIDEO_REPORT_DATA_RUNTIME_BUNDLE,
        'YACA': resource_types.YACA_REPORT_DATA_RUNTIME_BUNDLE,
        'NEWS': resource_types.NEWS_REPORT_DATA_RUNTIME_BUNDLE,
    }

    def get_data_resource_type_by_project(self):
        project_name = self.ctx.get(RDRI.Project.name)
        resource = self.DATA_RESOURCES_TYPE_BY_PROJECT.get(str(project_name).upper())
        if not resource:
            raise SandboxTaskFailureError('Incorrect project name, cannot find it in {}'.format(
                self.DATA_RESOURCES_TYPE_BY_PROJECT.keys()
            ))
        return resource

    def on_execute(self):
        # check project name before all actions
        result_resource_type = self.get_data_resource_type_by_project()
        logging.info('Try to create {} resource with report data.runtime'.format(result_resource_type))

        if 'subtask_ids' in self.ctx:
            task_res = self.process_subtask()
            # избавиться от одинаковых ресурсов
            uniq = self.uniq_task(task_res)
            logging.info("task_res=%s" % task_res)
            logging.info("uniq=%s" % uniq)
            # удалить дубли тасков и ресурсов
            # TODO временно отключаем удаление дублей. Потому что беты перестают обновляться. Нужна правка upperconf.pl
            # и кронтаба по обновлению data.runtime
            # self.delete_duplicate(task_res, uniq)
            # создать банл
            self.make_bundle(task_res, uniq)
        else:
            self.make_children()

    def make_children(self):
        if not (self.ctx[NumberBranches.name] > 0 or self.ctx[NumberTags.name] > 0):
            raise SandboxTaskFailureError(
                "one parametr must be positive. We have %s:%s %s:%s" %
                (NumberBranches.name, self.ctx[NumberBranches.name], NumberTags.name, self.ctx[NumberTags.name])
            )
        if self.ctx[CreateBundle.name] and self.ctx[NumberTags.name] <= 0:
            raise SandboxTaskFailureError('Bundle can create only from tags')

        # выставляем недостающие параметры
        if not self.ctx.get(ApacheBundleParameter.name):
            res = apihelpers.get_last_released_resource(ApacheBundleParameter.resource_type)
            if not res:
                raise SandboxTaskFailureError("Can not find last stable resource %s" % ApacheBundleParameter.resource_type)
            self.ctx[ApacheBundleParameter.name] = res.id

        project = self.ctx[RDRI.Project.name]
        task_ctx = []
        # получить список бранчей
        if self.ctx[NumberBranches.name] > 0:
            branches = self.report_branches(projects=[project], limit=self.ctx[NumberBranches.name])
            for url in branches[project]:
                ctx = {
                    RDRI.ProdSelector.name: 'beta',
                    RDRI.Project.name: project,
                    RDRI.Selector.name: 'svn',
                    Unit.ArcadiaUrl.name: os.path.join(url, 'arcadia/web/report'),
                    RDRI.UseExpiredSource.name: True,
                    ApacheBundleParameter.name: self.ctx[ApacheBundleParameter.name]
                }
                task_ctx.append((ctx, 0))

        # получить список тегов
        if self.ctx[NumberTags.name] > 0:
            tags = self.report_tags(projects=[project], limit=self.ctx[NumberTags.name], sublimit=self.ctx[NumberMinorTags.name])
            for url in tags[project]:
                ctx = {
                    RDRI.ProdSelector.name: 'production',
                    RDRI.Project.name: project,
                    RDRI.Selector.name: 'svn',
                    Unit.ArcadiaUrl.name: os.path.join(url, 'arcadia/web/report'),
                    RDRI.UseExpiredSource.name: True,
                    ApacheBundleParameter.name: self.ctx[ApacheBundleParameter.name]
                }
                task_ctx.append((ctx, 1))
        self.modify_task_ctx(task_ctx)

        subtask_id = []
        bundle_id = []
        for (ctx, is_tag) in task_ctx:
            # не делаем уведомления для дочерних задач
            ctx['notify_via'] = ''
            ctx["notifications"] = []
            subtask = self.create_subtask(
                task_type=RDR.ReportDataRuntime.type,
                description='Auto generate data.runtime (%s)' % ctx[Unit.ArcadiaUrl.name],
                input_parameters=ctx,
                important=self.important
            )
            subtask_id.append(subtask.id)
            if is_tag:
                bundle_id.append(subtask.id)

        self.ctx['subtask_ids'] = subtask_id
        self.ctx['bundle_ids'] = bundle_id
        channel.sandbox.server.wait_time(self.id, RDR.ReportDataRuntime.time_to_wait + 120)
        self.wait_all_tasks_completed(subtask_id)

    def process_subtask(self):
        error = []
        if 'deleted_task' not in self.ctx:
            self.ctx['deleted_task'] = {}

        task_res = {}
        # проверить что таски корректно отработали
        for sid in self.ctx['subtask_ids']:
            subtask = channel.sandbox.get_task(sid)
            if not subtask.is_finished() and subtask.id not in self.ctx['deleted_task']:
                error.append('Subtask {0} failed with status: {1}.'.format(subtask.id, subtask.status))
            else:
                res = channel.sandbox.get_resource(subtask.ctx['data_resource_id'])
                # удалить лишние таски и ресурсы, группируем по configuration и revision
                key = (res.attributes['configuration'], res.attributes['revision'])
                task_res[subtask.id] = {"task": subtask, "res": res, "key": key}
        if error:
            raise SandboxTaskFailureError('\n'.join(error))

        return task_res

    def arc_dir_listing(self, path, limit=1, filtered=None, sort=None):
        '''
          Получает листинг директории в репозитории, фильтрует и сортирует
        '''

        if limit < 0:
            limit = 1
        if limit == 0:
            limit = None

        items = Arcadia.list(path, as_list=True)

        if filtered:
            items = filter(filtered, items)
        if sort:
            sort(items)

        return [os.path.join(path, p) for p in items[:limit]]

    def _filter(self, regexp):
        return lambda(x): regexp.search(x)

    def _sort(self, regexp):
        def get_key(x):
            result = regexp.search(x)
            return [int(i) for i in result.groups(-1)] if result else [-1]
        return lambda(items): items.sort(cmp=lambda x, y: cmp(x, y), key=get_key, reverse=True)

    def _report_dirs(self, mode=None, projects=None, limit=1):
        if projects is None:
            projects = project_list
        if mode not in modes:
            mode = modes[0]

        rv = {}
        arc_url = Arcadia.tag_url if mode == 'tag' else Arcadia.branch_url
        for project_name in projects:
            # т.к. Arcadia.branch_url/Arcadia.tag_url добавляет к uri arcadia, то нужно указать path=..
            path = arc_url(regex[mode][project_name]['path'], '../')
            regexp = regex[mode][project_name]['regexp']
            rv[project_name] = self.arc_dir_listing(path, filtered=self._filter(regexp), sort=self._sort(regexp), limit=limit)
        return rv

    def report_tags(self, projects=None, limit=1, sublimit=0):
        mode = modes[1]
        tags = self._report_dirs(mode, projects, limit)
        # для тегов ограничить минорные версии
        for project_name in tags.keys():
            major = tags.pop(project_name)
            tags[project_name] = []
            for path in major:
                regexp = regex[mode][project_name]['regexp_minor']
                tags[project_name].extend(self.arc_dir_listing(path, filtered=self._filter(regexp), sort=self._sort(regexp), limit=sublimit))
        return tags

    def report_branches(self, projects=None, limit=1):
        return self._report_dirs(modes[0], projects, limit)

    def uniq_task(self, task_res):
        # сгруппировать по ключу
        expired = {}
        cache = {}

        for tid in task_res:
            subtask = task_res[tid]["task"]
            key = task_res[tid]["key"]

            # если ресурс был собран по просроченным источникам, то следует поискать ресурс собранный по свежим источникам
            if RDRI.IsExpired.name in subtask.ctx:
                if key not in expired:
                    expired[key] = subtask.id
            elif key not in cache:
                cache[key] = subtask.id

        # если все ресурсы собраны по просроченным источникам, то берем просроченный
        for key in expired:
            if key not in cache:
                cache[key] = expired[key]
        return cache

    def make_bundle(self, task_res, uniq):
        if not self.ctx[CreateBundle.name]:
            return

        resource_path = 'bundle-data.runtime'
        bundle = {}
        for sid in self.ctx['bundle_ids']:
            key = task_res[sid]["key"]
            if key not in uniq:
                raise SandboxTaskFailureError("Can not find key=%s in uniq" % key)

            tid = uniq[key]
            res = task_res[tid]["res"]
            logging.info("For task %s use resource %s from task %s" % (sid, res.id, tid))

            revision = res.attributes['revision']
            if not revision:
                logging.error("For task %s, resource=%s has not got attr revision" % (sid, res))
                # raise SandboxTaskFailureError("")
                continue

            bundle[res.id] = os.path.join(resource_path, revision)

        logging.info("bundle=%s" % bundle)

        def thread_expand(res_id, dest, error):
            try:
                Unit.TestReportUnit.expand_resource(res_id, dest)
            except Exception as e:
                logging.exception("error during expand_resource")
                error.append(res_id)
                raise e
        pool_error = []
        pool = [threading.Thread(target=thread_expand, args=(res_id, bundle[res_id], pool_error)) for res_id in bundle]
        map(threading.Thread.start, pool)
        map(threading.Thread.join, pool)
        if pool_error:
            raise SandboxTaskFailureError("Can not sync res_id: %s" % pool_error)
        logging.info("unpacked all data in the %s" % resource_path)

        if 'bundle_res' not in self.ctx:
            res_descr = "Bundle data.runtime for %s" % self.ctx[RDRI.Project.name]
            attrs = {
                'project': self.ctx[RDRI.Project.name],
                'configuration': 'production',
                'ttl': '7',
            }
            # для создания ресурса нужен относительный путь
            resource = self.create_resource(
                resource_type=self.get_data_resource_type_by_project(),
                resource_path=resource_path,
                description=res_descr,
                attributes=attrs
            )
            self.ctx['bundle_res'] = resource.id
            logging.info("Create resource %s" % resource)

    def delete_duplicate(self, task_res, uniq):
        delete_task = []
        delete_resource = []

        for sid in task_res:
            subtask = task_res[sid]["task"]
            res = task_res[sid]["res"]
            key = task_res[sid]["key"]
            if key not in uniq:
                raise SandboxTaskFailureError("Can not find key=%s in uniq" % key)

            if uniq[key] != subtask.id:
                delete_task.append(subtask.id)
                delete_resource.append(res.id)

        rest_client = common.rest.Client()
        if delete_resource:
            rest_client.batch.resources.delete = delete_resource
        logging.info("DELETE resources: %s" % delete_resource)

        if delete_task:
            rest_client.batch.tasks.delete = delete_task
            self.ctx['deleted_task'].update(dict.fromkeys(delete_task, 1))
        logging.info("DELETE tasks: %s" % delete_task)

    def mark_released_resources(self, status, ttl=28):
        return SandboxTask.mark_released_resources(self, status, ttl)

    def modify_task_ctx(self, task_ctx):
        pass


__Task__ = ReportDataRuntimeTags
