# -*- coding: utf-8 -*-

import logging
import os
import re
import zipfile

from sandbox.projects import resource_types

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.paths import add_write_permissions_for_path
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter

from sandbox.projects.common import apihelpers
import sandbox.projects.common.constants as consts


class WizdataPatchResource(LastReleasedResource):
    name = 'wizdata_patch_res_id'
    description = 'New data files (zip)'
    resource_type = resource_types.REPORTRULE_PATCH_ZIP
    required = False


class GztCompilerResource(LastReleasedResource):
    name = 'gztcompiler_res_id'
    description = 'Gztcompiler'
    resource_type = resource_types.WIZARD_DATA_COMPILER
    required = False


class RemorphCompilerResource(LastReleasedResource):
    name = 'remorphc_res_id'
    description = 'Remorph compiler'
    resource_type = resource_types.REMORPH_DATA_COMPILER
    required = False


class PrintwzrdResource(LastReleasedResource):
    name = 'printwzrd_res_id'
    description = 'Printwzrd'
    resource_type = resource_types.PRINTWZRD
    required = False


class PrintwzrdTestResource(LastReleasedResource):
    name = 'wizardreqs_res_id'
    description = 'Test for printwzrd'
    resource_type = resource_types.WIZARD_TEST_REQUESTS
    required = False


class CheckoutArcadiaURL(SandboxArcadiaUrlParameter):
    name = consts.ARCADIA_URL_KEY
    description = 'Path to svn (parent to "arcadia" folder)'
    default_value = 'arcadia:/arc/trunk/'
    required = False


class CheckoutArcadiaRevision(SandboxStringParameter):
    name = 'arcadia_revision'
    description = 'Arcadia revision'
    default_value = 'HEAD'
    required = False


class ReportRuleTestFull(SandboxTask):
    type = 'REPORT_RULE_TEST_FULL'

    arcadia_url = ''
    paths = dict()

    input_parameters = [
        WizdataPatchResource,
        GztCompilerResource,
        RemorphCompilerResource,
        PrintwzrdResource,
        CheckoutArcadiaURL,
        CheckoutArcadiaRevision,
        PrintwzrdTestResource,
    ]

    def get_arcadia_root_path(self):
        self.arcadia_url = self.ctx.get(consts.ARCADIA_URL_KEY)
        self.arcadia_revision = self.ctx.get('arcadia_revision')
        if self.arcadia_url and self.arcadia_url[-1] != '/':
            self.arcadia_url = self.arcadia_url + '/'
        if not self.arcadia_revision or self.arcadia_revision == 'HEAD':
            self.arcadia_revision = Arcadia.get_revision(self.arcadia_url)

    def checkout_dirs(self, dirs):
        for path in dirs:
            url = self.arcadia_url + path
            Arcadia.export(url, path, self.arcadia_revision)

    def apply_patch(self, datadir):
        datadir = os.path.join(self.abs_path(), datadir)

        self.get_resource('wizdata_patch', 'REPORTRULE_PATCH_ZIP')
        patchdir = os.path.join(self.paths['resources'], 'patch')
        make_folder(patchdir)
        add_write_permissions_for_path(patchdir)
        logging.info('Extracting archive to ' + patchdir)
        zipfile.ZipFile(self.paths['wizdata_patch']).extractall(patchdir)

        logging.info('Copy extracted file to data directory')

        gztsrcdir = os.path.join(datadir, 'gzt', 'src')
        rulesrcdir = os.path.join(datadir, 'rules', 'src')

        regzt = re.compile('\.data$')
        rerul = re.compile('\.rus$|\.tur$|\.ukr$|\.bel$|\.kaz$')

        names = os.listdir(patchdir)
        for name in names:
            srcname = os.path.join(patchdir, name)
            if os.path.isfile(srcname):
                if regzt.search(name):
                    copy_path(srcname, os.path.join(gztsrcdir, name))
                elif rerul.search(name):
                    copy_path(srcname, os.path.join(rulesrcdir, name))
                elif name == 'instruction':
                    copy_path(srcname, os.path.join(datadir, name))

    def get_resource(self, resname, restype):
        logging.info('Copy ' + resname + ' to tasks bin dir')
        if self.ctx[resname + '_res_id']:
            self.sync_resource(self.ctx[resname + '_res_id'])
            resource = channel.sandbox.get_resource(self.ctx[resname + '_res_id'])
        else:
            resource = apihelpers.get_last_resource(restype)
        resource_path = self.sync_resource(resource)

        self.paths['resources'] = os.path.join(self.abs_path(), 'resources')
        if not os.path.isdir(self.paths['resources']):
            make_folder(os.path.join(self.abs_path(), 'resources'))

        self.paths[resname] = os.path.join(self.paths['resources'], resname)

        if os.path.isfile(resource_path):
            copy_path(resource_path, self.paths[resname])
        else:
            raise SandboxTaskFailureError('Cannot find resource for ' + resname)

    def simplify_log(self, filein_path, fileout_path):
        filein = open(filein_path)
        fileout = open(fileout_path, 'w')
        s = ''
        for line in filein:
            if re.match('^src: ', line):
                if s:
                    fileout.write(s + "\n")
                s = line[5:].strip()
                s = re.sub('\$cgi\:lr\=', "\t", s)
                s = re.sub('\[|\]', '', s)
            if re.match('^Report.AcceptedRules', line):
                s = s + "\t" + line[21:].strip()
        filein.close()
        fileout.close()

    def on_execute(self):

        self.get_arcadia_root_path()

        logging.info('Start checkout arcadia folders...')
        wizard_source_dir = 'arcadia_tests_data/wizard_source_data/'
        wizard_runtime_dir = 'arcadia_tests_data/wizard.runtime/'
        wizard_dir = 'arcadia_tests_data/wizard/'

        self.checkout_dirs(dirs=[wizard_source_dir, wizard_dir, wizard_runtime_dir])
        self.checkout_dirs(dirs=['arcadia/search/wizard/', 'arcadia/kernel/', 'arcadia/tools/printwzrd/'])

        report_dir = wizard_source_dir + 'report/'

        if self.ctx['wizdata_patch_res_id']:
            logging.info('Applying patch to data')
            self.apply_patch(datadir=report_dir)

            logging.info('Creating source files')
            run_process([os.path.join(self.abs_path(), report_dir, 'bin', 'create_sources.sh')], log_prefix='create_sources')

            self.get_resource('gztcompiler', 'WIZARD_DATA_COMPILER')
            self.get_resource('remorphc', 'REMORPH_DATA_COMPILER')

            logging.info('Run gztcompiler')
            add_write_permissions_for_path(os.path.join(self.abs_path(), wizard_dir, 'gazetteer'))
            run_process(
                [
                    self.paths['gztcompiler'],
                    '-T', self.abs_path(),
                    os.path.join(self.abs_path(), wizard_dir, 'gazetteer/main.gzt'),
                ],
                log_prefix='gzt_compile'
            )

            logging.info('Run remorphc')
            rules_dir = os.path.join(self.abs_path(), wizard_dir, 'report')
            add_write_permissions_for_path(rules_dir)
            for lang in ('RUS', 'UKR', 'BEL', 'KAZ', 'TUR'):
                run_process(
                    [
                        self.paths['remorphc'],
                        '-t', 'config',
                        '-v', 'info',
                        os.path.join(self.abs_path(), wizard_source_dir, 'report/rules', lang, '.remorph'),
                    ],
                    log_prefix='remorphc'
                )

        self.get_resource('printwzrd', 'PRINTWZRD')
        self.get_resource('wizardreqs', 'WIZARD_TEST_REQUESTS')

        request_file = open(self.paths['wizardreqs'])
        full_log_resource = self.create_resource('Full printwzrd log', 'resources/full.printwzrd.log', 'PRINTWZRD_FULL_LOG')
        full_log = open(full_log_resource.path, 'w')

        logging.info('Run printwzrd with test requests')
        run_process(
            [
                self.paths['printwzrd'],
                '-a', os.path.join(self.abs_path(), 'arcadia_tests_data'),
                '-s', os.path.join(self.abs_path(), 'arcadia/tools/printwzrd/configs/wizard.cfg'),
                '-b'
            ],
            stdin=request_file,
            stdout=full_log,
            log_prefix='printwzrd'
        )
        logging.info('Stop printwzrd')
        full_log.close()

        logging.info('Simplifying lig file')
        simplified_log_resource = self.create_resource('Formatted printwzrd log', 'resources/simple.printwzrd.log', 'PRINTWZRD_SIMPLE_LOG')
        self.simplify_log(full_log_resource.path, simplified_log_resource.path)
        logging.info('Simplified log is ready')


__Task__ = ReportRuleTestFull
