# -*- coding: utf-8 -*-

import os
import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.common.share import ShareSandboxResource


class ResourcesForReshare(SandboxStringParameter):
    name = 'resources'
    description = 'Resources'


class ReshareShardsOnHost(SandboxTask):
    type = 'RESHARE_SHARDS_ON_HOST'

    input_parameters = (ResourcesForReshare,)

    def on_execute(self):
        resources = [int(r.strip()) for r in self.ctx['resources'].split(',')]
        for res_id in resources:
            resource = self._read_resource(res_id, sync=False)
            res_path = os.path.join(resource.abs_path())
            self.reshare_shard(res_path, res_id)

    def get_old_rbtorrent(self, res_path):
        old_rbtorrent = None
        shard_state_file = os.path.join(res_path, 'shard.state')
        if os.path.exists(shard_state_file):
            with open(shard_state_file) as shard_state:
                for line in shard_state:
                    if 'rbtorrent' in line:
                        old_rbtorrent = line.strip().replace('url:', '')
        return old_rbtorrent

    def get_files_list(self, shard_conf_file):
        logging.info(shard_conf_file)
        files = ['shard.conf', ]
        with open(shard_conf_file) as shard_conf:
            for line in shard_conf:
                if line.startswith('%attr'):
                    files.append(line.split()[-1])
        return files

    def reshare_shard(self, res_path, res_id):
        shard_conf_file = os.path.join(res_path, 'shard.conf')
        if not os.path.exists(shard_conf_file):
            raise SandboxTaskFailureError('Resource %s is not a shard' % res_id)
        files = self.get_files_list(shard_conf_file)
        old_rbtorrent = self.get_old_rbtorrent(res_path)
        new_rbtorrent = ShareSandboxResource(None, files, res_path).run()

        logging.info('sharing files: %s in %s' % (files, res_path))
        logging.info('new %s' % new_rbtorrent)
        logging.info('old %s' % old_rbtorrent)


__Task__ = ReshareShardsOnHost
