# -*- coding: utf-8 -*-

import logging

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters as sp

from sandbox.projects.release_machine import security as rm_sec
from sandbox.projects.common.search import resources_from_nanny
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import string
from sandbox.projects.common import utils
from sandbox.projects import resource_types

_RES_DATA = {
    "binary": {
        "checker": {"local_path": ["httpsearch"]},
        "possible_locations": [resources_from_nanny.SandboxFilesLocation],
        "res_type": resource_types.FRESH_RANKING_MIDDLESEARCH_EXECUTABLE
    },
    "models": {
        "checker": {"local_path": ["models.archive"]},
        "possible_locations": [resources_from_nanny.SandboxFilesLocation],
        "res_type": resource_types.DYNAMIC_MODELS_ARCHIVE
    },
    "config": {
        "checker": {"local_path": [
            "apache.ywsearch.cfg-a_geo_vla",
            "apache.ywsearch.cfg-a_geo_man",
            "apache.ywsearch.cfg-a_geo_msk",
            "apache.ywsearch.cfg-a_geo_sas",
        ]},
        "possible_locations": [resources_from_nanny.UrlLocation, resources_from_nanny.StaticFilesLocation],
        "res_type": resource_types.MIDDLESEARCH_CONFIG
    },
    "quick_rearr_data": {
        "checker": {"local_path": ["rearrange"]},
        "possible_locations": [resources_from_nanny.SandboxFilesLocation],
        "res_type": resource_types.QUICK_REARRANGE_RULES_DATA
    },
}
_RESULT_KEY = "loaded_resources_ids"


class ServiceId(sp.SandboxStringParameter):
    name = "service_id"
    description = "Service id"


class Attrs(sp.SandboxStringParameter):
    name = 'attributes'
    description = 'Additional attributes for resources (attr1=val1, attr2=val2 ...)'
    default_value = ''


class ResourcesToLoad(sp.SandboxBoolGroupParameter):
    name = "res_to_load"
    description = "Resources to load"
    choices = [(p, p) for p in _RES_DATA.iterkeys()]
    default_value = " ".join(_RES_DATA.iterkeys())


class ResourceLoaderFromNanny(SandboxTask):
    """
        Таск для скачивания ресурсов по шаблону из nanny
    """
    type = "RESOURCE_LOADER_FROM_NANNY"

    input_parameters = [ServiceId, ResourcesToLoad, Attrs]

    @property
    def footer(self):
        res_names = sorted(_RES_DATA.iterkeys())
        res_ids = []
        for n in res_names:
            res_id = self.ctx.get(_RESULT_KEY, {}).get(n)
            res_ids.append(lb.resource_link(res_id) if res_id else "-")
        return [{
            "helperName": "",
            "content": {
                "<h3>Loaded resources</h3>": {
                    "header": [
                        {"key": "res_name", "title": "Resource name"},
                        {"key": "res_id", "title": "Resource id"},
                    ],
                    "body": {
                        "res_name": res_names,
                        "res_id": res_ids
                    }
                }
            }
        }]

    def on_execute(self):
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=rm_sec.get_rm_token(self)
        )
        runtime_res_info = nanny_client.get_service_resources(self.ctx[ServiceId.name])
        logging.debug("Current resources info: %s", runtime_res_info)
        self.ctx[_RESULT_KEY] = {}

        for res_name, res_data in _RES_DATA.iteritems():
            if res_name not in utils.get_or_default(self.ctx, ResourcesToLoad):
                logging.info("Skip %s loading", res_name)
            else:
                for loc in res_data["possible_locations"]:
                    location = loc(res_name, res_data)
                    res_id = location.get_res_id(runtime_res_info)
                    if res_id:
                        self.ctx[_RESULT_KEY][res_name] = res_id
                        logging.info("Got resource '%s' = %s", res_name, res_id)
                        attrs = string.parse_attrs(utils.get_or_default(self.ctx, Attrs))
                        utils.set_resource_attributes(res_id, attrs)
                        break
                    else:
                        logging.info("Failed to get resource '%s' from location: %s", res_name, location.location)


__Task__ = ResourceLoaderFromNanny
