import logging
import tarfile
import os
from os.path import expanduser

import sandbox.sandboxsdk.task as sdk_task
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.sandboxapi import Sandbox


class Task(sdk_task.SandboxTask):
    """ An empty task, which does nothing except of greets you. """
    cores = 1
    type = "RESOURCES_TO_YT"

    def on_execute(self):
        self._provide_yt_token()
        resource_id = self._get_current_resource_id()
        binary_path = self._get_binary_path(resource_id)

        run_process([binary_path], log_prefix='resources_to_yt', wait=True)

    def _get_current_resource_id(self):
        client = Sandbox()
        resource_type = "RESOURCES_TO_YT_PACKAGE"
        resources = client.list_resources(resource_type=resource_type, order_by='-id', limit=1)
        logging.info([(resource.id, resource.timestamp) for resource in resources])
        if not resources:
            logging.error('there is not such resource')
        return resources[0].id

    def _provide_yt_token(self):
        yt_token = self.get_vault_data("TESTENGINE", "yt_token")
        home = expanduser("~")
        yt_folder = os.path.join(home, '.yt')
        os.mkdir(yt_folder)
        token_path = os.path.join(yt_folder, 'token')
        with open(token_path, 'w') as f:
            f.write(yt_token)

    def _get_binary_path(self, resource_id):
        folder = './'
        binary = self.sync_resource(resource_id)
        logging.info('Extracting %s to %s' % (binary, folder))
        tarfile.open(binary).extractall(folder)
        logging.info('files in ./ - ' + str(os.listdir(folder)))
        return os.path.join(folder, 'resources_to_yt', 'resources_to_yt')


__Task__ = Task
