import os
import errno
from subprocess import PIPE

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters as par
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process

import sandbox.common.types.resource as ctr
from sandbox.common.errors import TemporaryError

from sandbox.projects.common.apihelpers import get_last_resource_with_attribute
from sandbox.projects.sandbox import LXC_CONTAINER


class RunCubeStatus(SandboxTask):
    type = 'RUN_CUBE_STATUS'

    execution_space = 20000
    environment = (environments.PipEnvironment('PyYAML'),)

    class TaskContainer(par.Container):
        description = 'container'

        @property
        def default_value(self):
            container = get_last_resource_with_attribute(
                LXC_CONTAINER,
                attribute_name='run_cube_status',
                attribute_value='statinfra',
                status=ctr.State.READY,
            )
            return container.id

    class StepName(par.SandboxStringParameter):
        name = 'step_name'
        description = 'Step name'

    class EventParams(par.DictRepeater, par.SandboxStringParameter):
        name = 'event_params'
        description = 'Event params'
        default_value = ''

    input_parameters = [TaskContainer, StepName, EventParams]

    def on_execute(self):

        def mkdir_p(path):
            try:
                os.makedirs(path)
            except OSError as exc:  # Python >2.5
                if exc.errno == errno.EEXIST and os.path.isdir(path):
                    pass
                else:
                    raise

        os.chdir('/opt/work/etl/make/experimental/python/')
        import yaml
        ctx_config = yaml.load(open('/opt/work/etl/make/experimental/python/invoke.yaml'))

        file_with_yt_token = os.path.expanduser('~/.yt/token')
        mkdir_p(os.path.expanduser('~/.yt'))
        with open(file_with_yt_token, 'w') as f:
            f.write(ctx_config['yt']['token'])
        try:
            run_process(
                'invoke -e --hide=both {step} --date {date}'.format(
                    step=self.ctx[RunCubeStatus.StepName.name],
                    date=self.ctx[RunCubeStatus.EventParams.name]['date']
                ),
                shell=True, check=True, log_prefix='exec', stdin=PIPE,
                environment={
                    'YQL_TOKEN': ctx_config['yql']['token'],
                    'YT_TOKEN': ctx_config['yt']['token']
                }
            )
        except Exception as e:
            raise TemporaryError(e)


__Task__ = RunCubeStatus
