# -*- coding: utf-8 -*-

import sandbox.common.types.task as ctt

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.parameters import DictRepeater
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import utils

RETRIES = 10
DELAY = 30

TTL = 12 * 60 * 60  # 12 hours

ALEMATE_RUNNING_GROUP = ('NEW', 'COMMITTED', 'MERGED')
ALEMATE_FAILED_GROUP = ('REJECTED', 'FAILED')
ALEMATE_SUCCESS_GROUP = ('DONE')

DEPLOYMENT_TASKGROUP = 'deployment_taskgroup'
DEPLOYMENT_TASK = 'deployment_task'


class WaitDeployment(SandboxIntegerParameter):
    name = 'wait_deployment'
    description = 'Период опроса состояния автодеплоя (секунды)'
    default_value = 120


class NannyDashboardName(SandboxStringParameter):
    name = 'deployment_nanny_dashboard_name'
    description = 'Название дашборда в nanny для автодеплоя'
    required = True


class NannyDashboardRecipeName(SandboxStringParameter):
    name = 'deployment_nanny_dashboard_recipe'
    description = 'Название метарецепта'
    default_value = 'db_switch'
    required = True


class NannyDashboardItsTask(DictRepeater, SandboxStringParameter):
    name = 'deployment_nanny_dashboard_its_task'
    description = 'Словарь новых значений для ITS ручек'
    required = False


class NannyDashboardSetZkTask(DictRepeater, SandboxStringParameter):
    name = 'deployment_nanny_dashboard_set_zk_task'
    description = 'Словарь новых значений для ZK нод'
    required = False


class NannyWaitDeployParameter(SandboxBoolParameter):
    """
        Ждать завершения запущенного деплоя
    """
    name = 'deployment_nanny_bool_wait'
    description = 'Ждать завершения запущенного деплоя'
    default_value = False


class VaultName(SandboxStringParameter):
    name = 'vault_name'
    description = 'Sandbox vault c ID для Nanny'
    default_value = 'nanny-oauth-token'


class VaultOwner(SandboxStringParameter):
    name = 'vault_owner'
    description = 'Владелец sandbox vault'
    default_value = 'MEDIA_DEPLOY'


class SemaphoreName(SandboxStringParameter):
    name = 'semaphore_name'
    description = 'Семафор, лимитирующий одновременные выкатки'


class RunNannyDashboardRecipe(SandboxTask):
    """
        Запуск метарецепта дэшборда
    """
    type = 'RUN_NANNY_DASHBOARD_RECIPE'

    execution_space = 512

    cores = 1

    input_parameters = [
        NannyDashboardName,
        NannyDashboardRecipeName,
        NannyDashboardItsTask,
        NannyDashboardSetZkTask,
        NannyWaitDeployParameter,
        VaultName,
        VaultOwner,
        WaitDeployment,
        SemaphoreName,
    ]

    def initCtx(self):
        # set ttl
        self.ctx['kill_timeout'] = TTL

    @property
    def footer(self):
        headline = ''
        if 'request' in self.ctx:
            headline = headline + """
                <h4>Release requests: <a href="https://nanny.yandex-team.ru/ui/#/r/{request_id}/">{request_id}</a></h4>
            """.format(request_id=self.ctx['request']['id'])
        if DEPLOYMENT_TASK in self.ctx:
            headline = headline + """
                <h4>Deploy task: <a href="https://nanny.yandex-team.ru/ui/#/services/dashboards/catalog/{0}/deployments/catalog/{1}">{0}/</a></h4>
            """.format(self.ctx[NannyDashboardName.name], self.ctx[DEPLOYMENT_TASK])
        return headline

    def task_wait(self, task):
        if task.new_status not in list(
                self.Status.Group.FINISH +
                self.Status.Group.BREAK):
            self.wait_tasks(
                [task.id],
                list(self.Status.Group.FINISH + self.Status.Group.BREAK),
                wait_all=True
            )

    def on_enqueue(self):
        SandboxTask.on_enqueue(self)
        if self.ctx[SemaphoreName.name]:
            self.semaphores(ctt.Semaphores(
                acquires=[ctt.Semaphores.Acquire(name=self.ctx[SemaphoreName.name])],
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH)
            ))

    def on_execute(self):
        vault_key = utils.get_or_default(self.ctx, VaultName)
        vault_owner = utils.get_or_default(self.ctx, VaultOwner)
        wait_deployment = utils.get_or_default(self.ctx, WaitDeployment)
        zk_nodes = utils.get_or_default(self.ctx, NannyDashboardSetZkTask)
        its_values = utils.get_or_default(self.ctx, NannyDashboardItsTask)
        recipe_name = utils.get_or_default(self.ctx, NannyDashboardRecipeName)
        dashboard_name = utils.get_or_default(self.ctx, NannyDashboardName)

        token = self.get_vault_data(vault_owner, vault_key)
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=token,
        )

        if DEPLOYMENT_TASKGROUP not in self.ctx:
            # deploy
            self.set_info('Run meta recipe {0} for {1} dashboard'.format(recipe_name, dashboard_name))
            deployment = nanny_client.deploy_dashboard(
                dashboard_name,
                recipe_name,
                use_latest_snapshot=True,
                set_its_values=its_values,
                set_zk_values=zk_nodes
            )
            self.ctx[DEPLOYMENT_TASK] = deployment['_id']
            self.ctx[DEPLOYMENT_TASKGROUP] = deployment['taskgroup_id']

        if self.ctx[NannyWaitDeployParameter.name]:
            deploy_status = nanny_client.get_taskgroup_status(self.ctx[DEPLOYMENT_TASKGROUP])['status']
            if deploy_status in ALEMATE_RUNNING_GROUP:
                self.wait_time(wait_deployment)
            elif deploy_status in ALEMATE_FAILED_GROUP:
                raise SandboxTaskFailureError(
                    "Taskgroup {} is failed: {}".format(self.ctx[DEPLOYMENT_TASKGROUP], deploy_status)
                )
            elif deploy_status not in ALEMATE_SUCCESS_GROUP:
                raise Exception(
                    "Taskgroup {} has unknown state: {}".format(self.ctx[DEPLOYMENT_TASKGROUP], deploy_status)
                )


__Task__ = RunNannyDashboardRecipe
