# -*- coding: utf-8 -*-

import time
# import json
import random
import logging

from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxStringParameter, DictRepeater
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


logger = logging.getLogger(__name__)


class RecipeName(SandboxStringParameter):
    name = 'recipe_name'
    description = 'Nanny recipe name'
    required = True


class RecipeParams(DictRepeater, SandboxStringParameter):
    name = 'recipe_parameters'
    description = 'Dict of recipe params'
    required = True


class WaitExecution(SandboxBoolParameter):
    name = 'wait_task'
    description = 'Wait for a nanny task execution'
    default_value = True


class RunNannyRecipe(SandboxTask):
    """
        Запуск и ожидание выполнения рецепта в Няне.
    """

    execution_space = 128

    input_parameters = [RecipeName, RecipeParams, WaitExecution]

    type = 'RUN_NANNY_RECIPE'

    def on_execute(self):
        nanny_client = nanny.NannyClient(
            api_url='http://nanny.yandex-team.ru/',
            oauth_token=self.get_vault_data('MEDIA_DEPLOY', 'nanny-oauth-token')
        )

        if 'nanny_task_id' not in self.ctx:
            recipe_params = self.ctx.get('recipe_parameters') if self.ctx.get('recipe_parameters') else {}

            logging.debug("Recipe name: %s" % self.ctx.get('recipe_name'))
            logging.debug("Params are: %s" % recipe_params)

            logging.info("Creating nanny task")
            result = nanny_client.run_recipe(
                self.ctx.get('recipe_name'),
                recipe_params
            )

            self.ctx['nanny_task_id'] = result['id']
            self.set_info("Created Nanny task: <a href='https://nanny.yandex-team.ru/ui/#/alemate/taskgroups/{task}/table/'>{task}</a>".format(task=self.ctx['nanny_task_id']),
                          do_escape=False)
            logging.debug("Task id: %s" % self.ctx['nanny_task_id'])

        # Wait task execution
        while self.ctx.get('wait_task'):
            try:
                task_status = nanny_client.get_taskgroup_status(self.ctx['nanny_task_id'])['status']
                logging.debug("Task status: %s" % task_status)
            except Exception as e:
                logging.error("Got exception getting recipe status: %s" % e)
                time.sleep(random.randint(1, 10))
                continue

            if task_status == 'DONE':
                logging.debug("Task is done!")
                break
            elif task_status in ['MERGED', 'COMMITTED']:
                time.sleep(random.randint(10, 60))
            elif task_status in ['REJECTED']:
                raise SandboxTaskFailureError("Task rejected!")
            else:
                raise SandboxTaskFailureError('Unknown status: %s' % task_status)


__Task__ = RunNannyRecipe
