import os
import shutil
import logging
import yaml

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.parameters as sdk_parameters
import sandbox.sandboxsdk.sandboxapi as sdk_sandboxapi
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import apihelpers
from sandbox.projects import resource_types


def is_new_graph_launched(new_graphs_file):
    return os.path.getsize(new_graphs_file) > 2


def generate_launcher_config(sandbox_task, new_graphs_file, outfile):
    """
    Extract flowkeeper config from sandbox task
    :param sandbox_task:
    :return:
    """

    config = {"online_graph_launcher": {
        "nirvana": {},
        "sandbox": {}
    }}

    nirvana_config = config["online_graph_launcher"]["nirvana"]
    sandbox_config = config["online_graph_launcher"]["sandbox"]
    sandbox_config["new_graphs_file"] = new_graphs_file

    nirvana_config["oauth_token"] = sandbox_task.get_vault_data(
        sandbox_task.ctx[sandbox_task.TokenOwner.name],
        sandbox_task.ctx[sandbox_task.Token.name],
    )

    with open(outfile, 'w') as outf:
        yaml.dump(config, outf)


def get_latest_resource(resource_type, all_attrs=None):

    all_attrs = all_attrs or {}
    resources = channel.sandbox.list_resources(
        order_by="-id", limit=1, status="READY", resource_type=resource_type, all_attrs=all_attrs)

    if resources:
        return resources[0]
    logging.warning("Can't find latest resource: %s", resource_type)


def dump_previous_state_to_file(
    sandbox_task, filename,
    explicit_state_parameter, resource_type, attrs=None
):
    """
    :param sandbox_task:
    :param filename:
    :param explicit_state_parameter:
    :param resource_type:
    :param attrs:
    :return:
    """

    # in case state resource is is explicitly specified, use it
    state_resource_id = sandbox_task.ctx.get(explicit_state_parameter)

    if not state_resource_id:
        resource = get_latest_resource(
            resource_type=resource_type,
            all_attrs=attrs
        )
        if resource:
            state_resource_id = resource.id

    if not state_resource_id:
        raise ValueError

    shutil.copyfile(sandbox_task.sync_resource(state_resource_id), filename)


def save_state(sandbox_task, filename, resource_type, owner, new_graphs_launched):

    ml_task_id = sandbox_task.ctx[sandbox_task.MLTaskID.name]

    sandbox_task.create_resource(
        "Nirvana online learning state for ml_task_id %s" % ml_task_id,
        filename,
        resource_type,
        owner=owner,
        attributes={
            "ml_task_id": ml_task_id,
            "new_graphs_launched": new_graphs_launched,
            "is_processed": False
        }
    )


class RunNirvanaOnlineGraphLauncher(sdk_task.SandboxTask):
    cores = 1

    class NirvanaOnlineLearningNewGraphs(sdk_parameters.ResourceSelector):
        name = 'new_graphs_file'
        description = 'recently launched graphs, not processed yet'
        resource_type = resource_types.NIRVANA_ONLINE_LEARNING_NEW_GRAPHS
        required = False

    class MLTaskID(sdk_parameters.SandboxStringParameter):
        name = "ml_task_id"
        description = "ml task, used to retrieve state"
        required = True

    class FirstModel(sdk_parameters.SandboxStringParameter):
        name = "first_model"
        description = "YT path on hahn where first model is stores"
        required = True

    class NirvanaToken(sdk_parameters.SandboxStringParameter):
        name = "nirvana_oauth_token"
        description = "Nirvana OAuth token name (in sandbox vault)"
        required = False

    class NirvanaTokenOwner(sdk_parameters.SandboxStringParameter):
        name = "token_owner"
        description = "token owner"
        required = False

    class NirvanaURL(sdk_parameters.SandboxStringParameter):
        name = "nirvana_url"
        description = "Nirvana URL"
        required = False

    class TaskParamsConf(sdk_parameters.ResourceSelector):
        name = "task_yaml"
        description = "yaml with graph parameters"
        required = False

    class SetFirstModel(sdk_parameters.SandboxBoolParameter):
        name = "set_first_model"
        description = "whether to use first_model from learn_options and set it to first graph"
        required = True
        default_value = True

    class Stable(sdk_parameters.SandboxBoolParameter):
        name = "is_stable"
        description = "When true use stable binary from sandbox, when false use testing"
        default_value = True
        required = False

    type = "RUN_NIRVANA_ONLINE_GRAPH_LAUNCHER"

    input_parameters = [
        MLTaskID,
        NirvanaToken,
        NirvanaURL,
        NirvanaTokenOwner,
        TaskParamsConf,
        FirstModel,
        SetFirstModel,
        Stable
    ]

    def on_execute(self):

        new_graphs_file = "run_nirvana_online_learning_new_graphs_queue.json"

        task_id = self.ctx[self.MLTaskID.name]
        logging.info("Start graph launcher for task_id {task_id}".format(task_id=task_id))
        is_stable = self.ctx[self.Stable.name]
        release_status = sdk_sandboxapi.RELEASE_STABLE if is_stable else sdk_sandboxapi.RELEASE_TESTING
        build_task_id = apihelpers.get_last_released_resource(
            resource_types.ONLINE_GRAPH_LAUNCHER_BINARY,
            release_status
        )

        launcher_binary = self.sync_resource(build_task_id)

        logging.info("Working with launcher binary: {launcher_binary}".format(launcher_binary=launcher_binary))

        task_filename = self.sync_resource(
            self.ctx[self.TaskParamsConf.name]
        )

        prev_state_resource_id = apihelpers.get_last_resource_with_attrs(
            resource_type=resource_types.NIRVANA_ONLINE_LEARNING_NEW_GRAPHS,
            attrs={"ml_task_id": task_id, "new_graphs_launched": "True"},
            all_attrs=True
        )

        prev_state = None
        if prev_state_resource_id:
            prev_state = self.sync_resource(prev_state_resource_id)

        logging.info("Previous state file is {prev_state}".format(prev_state=prev_state))

        command = [
            launcher_binary,
            "--task", task_filename,
            "--new_graphs_file", new_graphs_file,

        ]
        if prev_state:
            command += ["--state_file_name", prev_state]
        else:
            # Check is last resource presented if not, add new model processing
            last_resource = apihelpers.get_last_resource_with_attrs(
                resource_type=resource_types.NIRVANA_ONLINE_LEARNING_NEW_GRAPHS,
                attrs={"ml_task_id": task_id}
                )
            if not last_resource and self.ctx[self.SetFirstModel.name]:
                command += ["--new", 'true']
                command += ["--first_model", self.ctx[self.FirstModel.name]]

        token = self.ctx.get(self.NirvanaToken.name)
        token_owner = self.ctx.get(self.NirvanaTokenOwner.name)

        if token and token_owner:
            oauth_token = self.get_vault_data(
                token_owner,
                token,
            )
        env = os.environ.copy()
        env['YT_TOKEN'] = oauth_token
        logging.info("Working with launcher binary: {launcher_binary}".format(launcher_binary=launcher_binary))
        sdk_process.run_process(
            command,
            wait=True,
            log_prefix='run_online_launcher',
            environment=env
        )

        logging.info(
            "Finish processing {task_id} state saved in file {new_graph_file}".format(
                task_id=task_id,
                new_graph_file=new_graphs_file
            )
        )

        save_state(
            self,
            new_graphs_file,
            resource_types.NIRVANA_ONLINE_LEARNING_NEW_GRAPHS,
            self.owner,
            is_new_graph_launched(new_graphs_file)
        )


__Task__ = RunNirvanaOnlineGraphLauncher
