# coding=utf-8

import logging
import time
import json
import uuid

from sandbox import common
from sandbox import sdk2

import requests
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


def str2bool(arg):
    if arg.lower() == 'true':
        return True
    if arg.lower() == 'false':
        return False
    return arg


class NirvanaClient(object):
    def __init__(self, oauth_token):
        self.url = 'https://nirvana.yandex-team.ru/api/public/v1/'
        self.oauth_token = oauth_token
        self.session = requests.Session()
        self.session.headers['Authorization'] = 'OAuth {}'.format(self.oauth_token)
        self.session.headers['Content-Type'] = 'application/json'
        retry = Retry(
            total=5,
            backoff_factor=0.3,
        )
        adapter = HTTPAdapter(max_retries=retry)
        self.session.mount('http://', adapter)
        self.session.mount('https://', adapter)

    def make_request(self, url, params):
        request_id = str(uuid.uuid4())
        jsondata = json.dumps({
            "jsonrpc": "2.0",
            "method": url,
            "id": request_id,
            "params": params
        })
        response = self.session.post(self.url + url, data=jsondata, verify=False)
        response.raise_for_status()
        response_content = response.json()
        print('Result: {}'.format(response_content))
        if 'result' not in response_content:
            if 'error' in response_content and 'message' in response_content['error']:
                raise Exception(str(response_content['error']['message']))
            else:
                raise Exception("Unknown exception")
        return response_content['result']


class RunNirvanaWorkflow(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        nirvana_token = sdk2.parameters.Vault("Vault with Nirvana Token", required=True)
        nirvana_quota = sdk2.parameters.String("Nirvana quota", default_value="default", required=True)

        nirvana_workflow_id = sdk2.parameters.String(
            "Template Nirvana workflow id",
            description="New workflow will be instantiated from this one",
            required=True
        )
        nirvana_global_options = sdk2.parameters.Dict("Nirvana workflow global options")
        stop_flow_on_terminate = sdk2.parameters.Bool("Stop Nirvana workflow on task termination")
        clone_to_new_workflow = sdk2.parameters.Bool(
            "Clone Nirvana workflow or not",
            description="Clone to new workflow instead of new instance in the same workflow",
            default=True
        )
        with clone_to_new_workflow.value[True]:
            nirvana_project_id = sdk2.parameters.String(
                "Project id",
                description="Will be given to the newly cloned workflow",
                required=False
            )
            nirvana_workflow_name = sdk2.parameters.String(
                "Workflow name",
                default_value="Sandbox driven workflow",
                description="Will be given to the newly cloned workflow",
                required=False
            )

        with sdk2.parameters.Output:
            executed_workflow_id = sdk2.parameters.String("Executed workflow id")
            executed_workflow_url = sdk2.parameters.Url("Executed workflow")

    def get_nirvana_client(self):
        nv_token = self.Parameters.nirvana_token.data()
        client = NirvanaClient(oauth_token=nv_token)
        return client

    @staticmethod
    def build_workflow_url(workflow_id):
        if workflow_id:
            return "https://nirvana.yandex-team.ru/flow/{}/graph".format(workflow_id)
        else:
            return None

    def on_execute(self):
        template_workflow_id = self.Parameters.nirvana_workflow_id
        template_workflow_url = RunNirvanaWorkflow.build_workflow_url(template_workflow_id)
        logging.info("Template workflow: %s", template_workflow_url)

        nirvana_workflow_name = self.Parameters.nirvana_workflow_name
        nirvana_project_id = self.Parameters.nirvana_project_id
        clone_to_new_workflow = self.Parameters.clone_to_new_workflow

        client = self.get_nirvana_client()

        exec_workflow_id = template_workflow_id
        if clone_to_new_workflow:
            exec_workflow_id = client.make_request("cloneWorkflow", dict(
                workflowId=template_workflow_id,
                newName=nirvana_workflow_name,
                newProjectCode=nirvana_project_id if nirvana_project_id else None,
                newQuotaProjectId=self.Parameters.nirvana_quota
            ))
        else:
            client.make_request("cloneWorkflowInstance", dict(
                workflowId=template_workflow_id,
                newQuotaProjectId=self.Parameters.nirvana_quota
            ))

        global_options = self.Parameters.nirvana_global_options
        if global_options:
            client.make_request("setGlobalParameters", dict(
                workflowId=exec_workflow_id,
                params=[dict(parameter=key, value=str2bool(value)) for key, value in global_options.items()]
            ))

        client.make_request("startWorkflow", dict(workflowId=exec_workflow_id))
        exec_workflow_url = RunNirvanaWorkflow.build_workflow_url(exec_workflow_id)

        self.Parameters.executed_workflow_url = exec_workflow_url
        self.Parameters.executed_workflow_id = exec_workflow_id

        message = "Starting workflow {}".format(exec_workflow_id)
        logging.info(message)

        need_to_fail = False
        while True:
            try:
                progress = dict(client.make_request("getExecutionState", dict(workflowId=exec_workflow_id)))
                logging.info("Workflow %s progress info: %s", exec_workflow_id, progress)
                if progress["status"] == "completed":
                    execution_result = progress["result"]
                    if execution_result != "success":
                        logging.info("Workflow completed with no success (result: %s)", execution_result)
                        need_to_fail = True
                    break
            except Exception as exc:
                logging.warning("Failed to execute GetExecutionState: %s", exc)
            time.sleep(60)

        if need_to_fail:
            if execution_result == "cancel":
                raise common.errors.TaskFailure("Workflow has been cancelled. See {}".format(exec_workflow_url))
            else:
                raise common.errors.TaskFailure("Workflow has been failed. See {}".format(exec_workflow_url))

    def on_timeout(self, prev_status):
        workflow_url = self.Parameters.executed_workflow_url
        logging.info("Task timed out, workflow URL: %s, executed_workflow_id: %s", workflow_url, self.Parameters.executed_workflow_id)

        if not self.Parameters.executed_workflow_id:
            raise Exception("Task timed out, but Parameters.executed_workflow_id is empty. Could not get running blocks information")
        else:
            self.set_info("Workflow {} had timed out.".format(workflow_url))

    def on_terminate(self):
        if not self.Parameters.stop_flow_on_terminate:
            return
        workflow_id = self.Parameters.executed_workflow_id
        self.get_nirvana_client().make_request("stopWorkflow", dict(workflowId=workflow_id))
        logging.info("Workflow has been stopped.")

    @property
    def footer(self):
        if self.Parameters.executed_workflow_url:
            href = str(self.Parameters.executed_workflow_url)
            return '<a href="{}">Executed workflow</a>'.format(href)
        else:
            href = RunNirvanaWorkflow.build_workflow_url(self.Parameters.nirvana_workflow_id)
            return 'Workflow not created. <a href="{}">Template workflow</a>'.format(href)
