# -*- coding: utf-8 -*-

import logging

from sandbox.projects.common import apihelpers
from sandbox.projects.common.fusion.task import (DISK_TEST_TYPE,
                                                 MEMORY_TEST_TYPE,
                                                 MIXED_TEST_TYPE)
from sandbox.projects.common.search import components as sc
from sandbox.projects.websearch.basesearch.PriemkaBasesearchBinary import \
    PriemkaBasesearchBinary
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask


Basesearch1Params = sc.create_fusion_params(1, old_task=True)
Basesearch2Params = sc.create_fusion_params(2, old_task=True)


def get_resource_for_tests(resource_type, attrs=None, fusion_test_attrs={"test_fusion": None}):
    attributes = {}
    attributes.update(fusion_test_attrs)
    if attrs:
        attributes.update(attrs)

    return apihelpers.get_last_resource_with_attrs(resource_type, attributes, all_attrs=True).id


class RunRefreshAcceptance(SandboxTask):
    type = 'RUN_REFRESH_ACCEPTANCE'

    input_parameters = (sc.NewFusionBinary, sc.OldFusionBinary)

    old_performance_test = "old_performance_test"
    new_performance_test = "new_performance_test"

    FRESHNESS_ATTR = "test_fusion_freshness"

    @property
    def footer(self):
        if not self.is_completed():
            return None
        foot = self._stats_footer()
        return foot

    def _stats_footer(self):
        head = [
            {"key": "rps_0", "title": "Old Refresh RPS"},
            {"key": "rps_1", "title": "New Refresh RPS"},
        ]
        return [
            {
                'helperName': '',
                'content': {
                    "<h4>Stats</h4>": {
                        "header": head,
                        "body": {
                            "rps_0":
                                [self.ctx['rps_0']],
                            "rps_1":
                                [self.ctx['rps_1']],
                        }
                    }
                }
            }
        ]

    def update_ctx_with_db_source_params(self, ctx, test_type, params):
        if test_type == DISK_TEST_TYPE:
            ctx.update({params.DbSource.name: sc.FUSION_DB_PRE_CREATED})
        else:
            ctx.update(
                {
                    sc.DefaultFusionParams.MaxDocs.name: 40000,
                }
            )
            if test_type == MEMORY_TEST_TYPE:
                pass  # enable distributor
            if test_type == MIXED_TEST_TYPE:
                pass  # enable distributor
        return ctx

    def enhance_with_fusion_params(self, ctx, fusion_executable_id, params=None, db=None, test_type=None, configs_attr=None):
        db = db or get_resource_for_tests(
            "SEARCH_DATABASE",
            {'test_refresh': '1'},
            {},
        )
        params = params or sc.DefaultFusionParams
        context = {
            params.OxygenConfig.name: get_resource_for_tests(
                "REFRESH_TESTING_CONFIGS",
                {'refresh_test_oxygen_config': "yes"},
                {},
            ),
            params.Binary.name: fusion_executable_id,
            params.BasesearchConfig.name: get_resource_for_tests(
                "REFRESH_TESTING_CONFIGS",
                {'refresh_test_basesearch_config': "yes"},
                {},
            ),
            params.Database.name: db,
            params.Config.name: get_resource_for_tests(
                "REFRESH_TESTING_CONFIGS",
                {'refresh_test_fusion_config': "yes"},
                {},
            ),
            params.Models.name: get_resource_for_tests(
                "DYNAMIC_MODELS_ARCHIVE",
                {'TE_web_production_mmeta_models': '07-03-16_20:04'},
                {}
            ),
            params.StaticData.name: get_resource_for_tests(
                "RTYSERVER_MODELS",
                None,
                {}
            ),
        }
        logging.debug('context_debug : {}'.format(context))
        if test_type:
            self.update_ctx_with_db_source_params(context, test_type, params)
        context.update(ctx)
        return context

    def create_performance_context(self, fusion_executable_id, test_type, configs_attr=None):
        sub_ctx = {
            # dolbiling params:
            'dolbilo_plan_resource_id': get_resource_for_tests(
                "BASESEARCH_PLAN",
                {'test_refresh': '1'},
                {},
            ),
            'dolbilka_executor_requests_limit': 100000,
            'dolbilka_executor_mode': 'finger',
            'dolbilka_executor_max_simultaneous_requests': 21,
            'dolbilka_executor_sessions': 5,
            'notify_via': '',
            'notify_if_failed': self.owner
        }

        return self.enhance_with_fusion_params(sub_ctx, fusion_executable_id, test_type=test_type, configs_attr=configs_attr)

    def run_performance_test(self, fusion_executable_id, postfix="", test_type="", configs_attr=None):
        context = self.create_performance_context(
            fusion_executable_id, test_type, configs_attr)
        logging.info(context)
        return self.create_subtask(
            task_type="TEST_FUSION_PERFORMANCE_BEST",
            description="[{0}] Performance task for #{1}:{2} {3}".format(
                postfix,
                self.id,
                self.descr,
                test_type,
            ),
            input_parameters=context,
            model=PriemkaBasesearchBinary.default_performance_task_cpu_model['linux'],
            important=self.important,
            arch="linux"
        ).id

    def get_run_function(self, task_name):
        tasks_map = {
            self.old_performance_test: lambda: self.run_performance_test(self.ctx[sc.OldFusionBinary.name], "Old", DISK_TEST_TYPE),
            self.new_performance_test: lambda: self.run_performance_test(self.ctx[sc.NewFusionBinary.name], "New", DISK_TEST_TYPE),
        }

        return tasks_map[task_name]

    def run_child_task(self, task_name):
        if not self.ctx.get(task_name):
            logging.info("Launching task %s", task_name)
            self.ctx[task_name] = self.get_run_function(task_name)()
            self.ctx["wait_tasks"].append(self.ctx[task_name])
            logging.info("Task id: %s", self.ctx[task_name])

    def on_execute(self):
        self.ctx["wait_tasks"] = []
        tasks = [
            # order matters!
            self.old_performance_test,
            self.new_performance_test,
        ]

        for task in tasks:
            self.run_child_task(task)

        not_ok_statuses = tuple(self.Status.Group.FINISH) + \
            tuple(self.Status.Group.BREAK)
        running_tasks = filter(
            lambda t: t.status not in not_ok_statuses, self.list_subtasks(load=True))

        if running_tasks:
            self.wait_all_tasks_stop_executing(running_tasks)

        result_1 = self.ctx["old_performance_test"]
        result_2 = self.ctx["new_performance_test"]
        logging.debug('CONTEX_LOOK_HERE!!!')
        logging.debug(channel.sandbox.get_task(result_1).ctx)
        self.ctx['rps_0'] = channel.sandbox.get_task(
            result_1).ctx['requests_per_sec']
        self.ctx['rps_1'] = channel.sandbox.get_task(
            result_2).ctx['requests_per_sec']
        logging.debug('rps_0 = {} rps_1 = {}'.format(
            self.ctx['rps_0'], self.ctx['rps_1']))


__Task__ = RunRefreshAcceptance
