import os
import json
import re
import types
import shutil
import logging
import base64
from pprint import pformat
import glob
import sys
from collections import OrderedDict
import pkg_resources
#import cPickle as pickle

from sandbox.projects import resource_types as rt
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process as do_run_process
from sandbox.sandboxsdk import parameters
#from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class RuntimeErrorWithCode(RuntimeError):
    pass


def reraise(code):
    t, e, tb = sys.exc_info()
    if isinstance(e, RuntimeErrorWithCode):
        raise t, e, tb
    raise RuntimeErrorWithCode, RuntimeErrorWithCode(code, e), tb


def run_process(*args, **kwargs):
    if 'close_fds' not in kwargs:
        kwargs['close_fds'] = True
    do_run_process(*args, **kwargs)


def extract_python_wheel(archive, dest_dir):
    run_process(['unzip', '-qq', '-d', dest_dir, '-x', archive])


def _parse_entry_points(file):
    with open(file) as ep_file:
        ep_map = pkg_resources.EntryPoint.parse_map(ep_file.read())

    exports = OrderedDict()
    for group, items in sorted(ep_map.items()):
        exports[group] = OrderedDict()
        for item in sorted(map(str, items.values())):
            name, export = item.split(' = ', 1)
            exports[group][name] = export

    return exports


def create_console_scripts(entry_points_file, bin_root):
    exports = _parse_entry_points(entry_points_file)
    console_scripts = exports.get('console_scripts')
    if not console_scripts:
        return

    for name, descr in console_scripts.items():
        module, function = descr.split(':')

        bin_path = os.path.join(bin_root, name)
        with open(bin_path, 'w') as bin:
            print >>bin, '''#! /usr/bin/env python

from {module} import {function}

if __name__ == '__main__':
    {function}()'''.format(module=module, function=function)

        os.chmod(bin_path, 0755)


class RunRemJobPacket(SandboxTask):
    """ REM job-packet executor """

    type = 'RUN_REM_JOBPACKET'

    cores = 1

    LIBRARY_TYPES = frozenset(['py', 'ld', 'bin'])

    class PacketId(parameters.SandboxStringParameter):
        name = "pck_id"
        description = "pck_id"
        required = False

    class Executor(parameters.ResourceSelector):
        name = "executor_resource"
        description = "executor_resource"
        resource_type = rt.REM_JOBPACKET_EXECUTOR
        required = True

    class ExecutionSnapshotData(parameters.SandboxStringParameter):
        name = "snapshot_data"
        description = "snapshot_data"
        required = False
        multiline = True

    class ExecutionSnapshotResource(parameters.ResourceSelector):
        name = "snapshot_resource_id"
        description = "snapshot_resource_id"
        resource_type = rt.REM_JOBPACKET_EXECUTION_SNAPSHOT
        required = False

    class RemServerAddr(parameters.SandboxStringParameter):
        name = "rem_server_addr"
        description = "rem_server_addr"
        required = True

    class CustomResourcesDescr(parameters.SandboxStringParameter):
        name = "custom_resources"
        multiline = True
        description = "custom resources"
        required = False

    # For cache-locality of resources
    class CustomResources(parameters.ResourceSelector):
        name = "custom_resources_list"
        resource_type = None
        description = "custom resources"
        multiple = True
        required = False

    class PythonVirtualEnvironment(parameters.ResourceSelector):
        name = "python_resource"
        description = "python"
        resource_type = rt.REM_JOBPACKET_PYTHON
        required = True

    class ResumeParams(parameters.SandboxStringParameter):
        name = "resume_params"
        description = "resume params"
        required = False

    class VaultsSetup(parameters.SandboxStringParameter):
        name = "vaults_setup"
        description = "vaults_setup"
        required = False

    input_parameters = [
        RemServerAddr,
        PacketId,
        Executor,
        ResumeParams,
        ExecutionSnapshotData,
        ExecutionSnapshotResource,
        CustomResources,
        CustomResourcesDescr,
        PythonVirtualEnvironment,
        VaultsSetup,
    ]

    def arcadia_info(self):
        return '', None, 1

    @classmethod
    def __parse_custom_resources_any(cls, resources):
        if isinstance(resources, types.StringTypes):
            resources = resources.strip()

            if resources:
                if resources[0] == '=':
                    resources = resources[1:]
                return list(cls.__parse_custom_resources(json.loads(resources)))
            else:
                return None

        elif isinstance(resources, dict):
            return list(cls.__parse_custom_resources(resources))

        else:
            return None

    @staticmethod
    def __parse_custom_resources(raw_resources):
        for local_name, full_path in raw_resources.items():
            m = re.match('^(?:sbx:)(\d+)(/.*)?$', full_path)
            if not m:
                raise RuntimeErrorWithCode('E_MALFORMED_RESOURCE_PATH', None)

            resource_id, in_resource_path = m.groups()

            yield int(resource_id), \
                  in_resource_path or None, \
                  local_name
                  #None if in_resource_path and '*' in in_resource_path else local_name

    def __sync_custom_resources(self, file_map):
        logging.debug('__sync_custom_resources(%s)' % file_map)

        custom_resources_ids = {
            resource_id
                for resource_id, in_content_path, local_name in file_map
        }

        logging.debug('resource_ids: %s' % custom_resources_ids)

        resources = {}

        for resource_id in custom_resources_ids:
            try:
                resources[resource_id] = self.sync_resource(resource_id)
            except:
                # XXX Not always permanent error!
                # 2016-06-24Task error: Cannot download resource #141333473. Error: 1.
                reraise('E_CUSTOM_RESOURCE_SYNC_FAILED')

        logging.debug('resource: %s' % resources)

        landing_folder = 'custom_resources'
        target_prefix = 'work/root'

        symlinks = set()
        libraries_by_type = {lib_type: [] for lib_type in self.LIBRARY_TYPES}

        def symlink(target, name):
            if os.path.lexists(name):
                os.unlink(name)  # TODO Be more strict: raise

            logging.debug('user.symlink(%s, %s)' % (target, name))
            os.symlink(target, name)
            symlinks.add(name)

        def symlink_lib(target, name):
            logging.debug('lib.symlink(%s, %s)' % (target, name))
            os.symlink(target, name)

        def process_as_library():
            if len(mapping) > 1:
                raise RuntimeErrorWithCode(
                    'E_MULTIPLE_PYTHON_MODULES',
                    '')

            library_path, matched_name, orig_name = mapping[0]

            logging.debug('%s-library.0 %s' % (
                lib_type, (local_name, library_path, matched_name, orig_name),))

            if os.path.isdir(library_path):
                libraries.append(os.path.abspath(library_path))
            else:
                # TODO no check for same name

                real_local_name = local_name[len(lib_type) + 1:]

                if real_local_name.startswith('@'):
                    link_name = orig_name or os.path.basename(library_path)
                    assert link_name
                else:
                    link_name = real_local_name

                logging.debug('%s-library.1 %s <= %s' % (lib_type, library_path, link_name))
                symlink_lib('../' + library_path, lib_type + 'lib/' + link_name)

        for resource_id, in_content_path, local_name in file_map:
            resource_path = resources[resource_id]
            logging.debug('iter -> %s' % ((resource_id, in_content_path, local_name),))

            if in_content_path:
                if os.path.isdir(resource_path):
                    content_path = landing_folder + '/%s' % resource_id

                    logging.debug('treat as folder with content')
                    logging.debug('os.symlink(%s, %s)' % (resource_path, content_path))

                    if not os.path.lexists(content_path):
                        os.symlink(resource_path, content_path)

                elif any(resource_path.endswith(ext) for ext in ['.tar', '.tar.gz', '.tgz', '.whl']):
                    content_path = landing_folder + '/%s_x' % resource_id

                    logging.debug('treat as archive with content')
                    logging.debug('os.mkdir(%s)' % content_path)

                    if not os.path.exists(content_path):
                        os.mkdir(content_path)

                        if resource_path.endswith('.whl'):
                            extract_python_wheel(resource_path, content_path)
                        else:
                            run_process(['tar', '-C', content_path, '-xf', resource_path])

                else:
                    raise ValueError()

                # means that user treat first element in path as directory or some archive (deb)
                first_dir_match = re.match(r'^/*([^/]+)(/.*)$', in_content_path)

                if first_dir_match:
                    first_node, rest_path = first_dir_match.groups()
                    logging.info('Matched first dir: %s and %s', first_node, rest_path)
                    matched_first_node_files = glob.glob(content_path + '/' + first_node)
                    logging.info('glob(%s) => %s', content_path + '/' + first_node, matched_first_node_files)
                    if matched_first_node_files:
                        contents = []
                        for file in matched_first_node_files:
                            logging.info('File: %s', file)
                            if not os.path.isfile(file):
                                logging.info('Not file(%s). Add content: %s, %s', file, '', file + rest_path)
                                contents += [('', file + rest_path)]
                                continue

                            new_content_path = landing_folder + '/%s_x_%s' % (resource_id, os.path.basename(file))
                            new_temp_path = landing_folder + '/%s_x_temp_%s' % (resource_id, os.path.basename(file))
                            logging.info('new content path: %s', new_content_path)
                            if not os.path.exists(new_content_path):
                                logging.info('mkdir(%s)', new_content_path)
                                os.mkdir(new_content_path)
                                if not os.path.exists(new_temp_path):
                                    logging.info('mkdir(%s) for temp', new_temp_path)
                                    os.mkdir(new_temp_path)

                                if file.endswith('.deb'):
                                    self._extract_debian_package(file, new_content_path, temp_path=new_temp_path)
                                else:
                                    raise RuntimeErrorWithCode(
                                        'E_UNKNOWN_ARCHIVE_FORMAT',
                                        "Archive file with unknown format %s in resource %d" % (file, resource_id)
                                    )

                            logging.info('Add content: %s, %s', new_content_path, rest_path)
                            contents += [(new_content_path, rest_path)]
                    else:
                        logging.info('Add content (will fail below): %s, %s', content_path, in_content_path)
                        contents = [(content_path, in_content_path)]
                else:
                    logging.info('First dir not matched. Add content: %s, %s', content_path, in_content_path)
                    contents = [(content_path, in_content_path)]

                mapping = []

                for content_path, in_content_path in contents:
                    full_path = content_path + in_content_path

                    matched_files = glob.glob(full_path)
                    logging.debug('glob(%s) => %s' % (full_path, matched_files))

                    if os.path.exists(full_path) or os.path.lexists(full_path):
                        mapping += [
                            # ('custom_resources/82342/geobase3.so', 'py:geobase', None)
                            # ('custom_resources/82342/lib', 'ld:geobase', None)
                            # ('custom_resources/82342_x/.../zklock.py', 'zklock.py', None)
                            (full_path, local_name, None)
                        ]

                    elif matched_files:
                        mapping += [
                            # ('custom_resources/82342/geobase3.so', 'geobase3.so', None)
                            # ('custom_resources/82342_x/.../zklock.py', 'zklock.py', None)
                            # ('custom_resources/82342_x/.../libX', 'libX', None)
                            (file, os.path.basename(file), None) for file in matched_files
                        ]

                    else:
                        raise RuntimeErrorWithCode(
                            'E_NO_FILE_IN_RESOURCE',
                            "No '%s' file in resource %d" % (in_content_path, resource_id))

            else:
                landing_path = landing_folder + '/%d' % resource_id
                if not os.path.lexists(landing_path):
                    os.symlink(resource_path, landing_path)

                mapping = [
                    # ('custom_resources/82342', 'py:geobase3', 'geobase3.so')
                    # ('custom_resources/82342', 'geobase3.so', 'geobase3.so')
                    # ('custom_resources/82342', 'ld:geobase', 'lib')
                    (landing_path, local_name, os.path.basename(resource_path))
                ]

            if local_name.startswith('whl:'):
                if in_content_path:
                    raise RuntimeError("whl: and in-resource-path are mutually exclusive")

                content_path = landing_folder + '/%s_x' % resource_id  # TODO copy-pasted
                if not os.path.exists(content_path):
                    os.mkdir(content_path)
                    extract_python_wheel(resource_path, content_path)

                # 1. (actually only top_levels.txt is need to be in PYTHONPATH)
                libraries_by_type['py'].append(os.path.abspath(content_path))

                # 2. create entry_points
                def get_dist_info_dir(root):
                    dirs = glob.glob(root + '/*.dist-info')
                    if not dirs:
                        raise RuntimeError("Can't find .dist-info directory for %d" % resource_id)
                    elif len(dirs) > 1:
                        raise RuntimeError("Several .dist-info directories in %d: %s" % (
                            resource_id, ', '.join(dirs)))
                    return dirs[0]

                dist_info_dir = get_dist_info_dir(content_path)
                entry_points_file = dist_info_dir + '/entry_points.txt'

                if os.path.exists(entry_points_file):
                    entry_points_bins_path = landing_folder + '/%s_entry_points' % resource_id
                    if not os.path.exists(entry_points_bins_path):  # j.i.c
                        os.mkdir(entry_points_bins_path)
                        create_console_scripts(entry_points_file, entry_points_bins_path)
                        libraries_by_type['bin'].append(os.path.abspath(entry_points_bins_path))

            else:
                libraries = None
                for lib_type in self.LIBRARY_TYPES:
                    if local_name.startswith(lib_type + ':'):
                        libraries = libraries_by_type[lib_type]
                        break

                if libraries is not None:
                    process_as_library()

                else:
                    for target, name, _ in mapping:
                        symlink('../../' + target, target_prefix + '/' + name)

        return list(symlinks), libraries_by_type

    def _extract_debian_package(self, debian_package, content_path, temp_path):
        logging.info('Extracting debian package %s to %s, temp dir %s', debian_package, content_path, temp_path)
        debian_package_abs = os.path.abspath(debian_package)
        logging.debug('debian_package(%s)', debian_package_abs)
        content_path_abs = os.path.abspath(content_path)

        prev_dir = os.getcwd()  # Workaround with chdir, `ar` can't output to the specified directory
        logging.debug('cwd(%s)', prev_dir)
        logging.debug('chdir(%s)', temp_path)
        os.chdir(temp_path)

        try:
            logging.debug('cwd(%s)', os.getcwd())
            logging.debug('debian %s file exists=%s', debian_package_abs, os.path.exists(debian_package_abs))
            run_process(['ar', 'x', debian_package_abs])
            logging.info('temp path glob(%s) => %s', os.getcwd(), glob.glob('*'))

            UNTAR_EXTENSIONS = [
                ('.gz', '-xf'),
                ('.xz', '-xJf'),
            ]

            untar_filename = None
            untar_flag = None
            for ext, flag in UNTAR_EXTENSIONS:
                filename = 'data.tar' + ext

                file_exists = os.path.exists(filename)
                logging.debug('exists(%s)=%s', filename, file_exists)

                if file_exists:
                    untar_filename = filename
                    untar_flag = flag

            if not untar_flag:
                raise ValueError("Unsupported debian package compression")

            run_process(['tar', untar_flag, untar_filename, '-C', content_path_abs])
            logging.info('content path glob(%s) => %s', content_path_abs, glob.glob(content_path_abs + '/*'))
        finally:
            logging.debug('chdir(%s)', prev_dir)
            os.chdir(prev_dir)

    @property
    def __custom_resources(self):
        return self.ctx['_custom_resources_parsed']

    def _produce_vaults_config(self, orig_setups, vaults_dir):
        # XXX Stupid implementation ever
        def get_vault_data(id):
            if isinstance(id, (tuple, list)):
                return self.get_vault_data(*id)
            return self.get_vault_data(id)
            # return 'value-for-%s' % (id,)

        def produce(prefix, (vault_files, vault_vars)):
            def expand_vaults(pairs):
                return [
                    (env_var, get_vault_data(vault_id)) for (env_var, vault_id) in pairs
                ]

            if vault_files is not None:
                vault_files = expand_vaults(vault_files)
            if vault_vars is not None:
                vault_vars = expand_vaults(vault_vars)

            def out(env_var, value):
                filename = vaults_dir + '/' + prefix + '-' + env_var.lower()
                with open(filename, 'w') as out:
                    print >>out, value
                return filename

            if vault_files is not None:
                vault_files = [
                    (env_var, out(env_var, vault_value))
                        for (env_var, vault_value) in vault_files
                ]

            return (vault_files or []) + (vault_vars or [])

        return {
            'global': produce('global', orig_setups['global']),
            'jobs': {
                job_id: produce(job_id, setup)
                    for job_id, setup in orig_setups['jobs'].items()
            }
        }

    def __do_on_execute(self):
        logging.debug("on_execute work dir: %s" % os.getcwd())

        logging.debug(pformat(self.ctx))
        logging.debug(type(self.ctx['custom_resources']))

        try:
            python_virtual_env_path = self.sync_resource(int(self.ctx['python_resource']))
        except:
            reraise('E_PYTHON_RESOURCE_SYNC_FAILED')

        os.mkdir('python')
        if python_virtual_env_path.endswith('.tar') or python_virtual_env_path.endswith('.tar.gz'):
            run_process(['tar', '-C', 'python', '-xf', python_virtual_env_path])
        elif python_virtual_env_path.endswith('/python'):
            os.mkdir('python/bin')
            os.symlink(python_virtual_env_path, 'python/bin/python')
        else:
            raise RuntimeError("Unknown REM_JOBPACKET_PYTHON format")
        python_virtual_env_bin_directory = os.path.abspath('./python/bin')

        # TODO use lib/* hierarchy
        for lib_type in self.LIBRARY_TYPES:
            os.mkdir(lib_type + 'lib')

        os.mkdir('custom_resources')
        os.mkdir('work')

        prev_packet_snapshot_file = None
        if self.ctx['snapshot_resource_id']:
            try:
                prev_snapshot_path = self.sync_resource(int(self.ctx['snapshot_resource_id']))
            except:
                reraise('E_SNAPSHOT_RESOURCE_SYNC_FAILED')

            if False:
                for subdir in ['io', 'root']:
                    shutil.copytree(
                        prev_snapshot_path + '/' + subdir,
                        'work/' + subdir,
                        symlinks=True
                    )
                prev_packet_snapshot_file = prev_snapshot_path + '/' + 'packet.pickle'

            else:
                run_process(['tar', '-C', 'work', '-xf', prev_snapshot_path])
                prev_packet_pickle_basename = 'prev_packet.pickle'
                os.rename('work/packet.pickle', prev_packet_pickle_basename)
                prev_packet_snapshot_file = prev_packet_pickle_basename

        # snapshot_data may be E2BIG for execve
        elif self.ctx['snapshot_data']:
        # TODO DRY
            prev_packet_snapshot_file = 'initial_snapshot.pickle'

            with open(prev_packet_snapshot_file, 'w') as out:
                out.write(
                    base64.b64decode(
                        self.ctx['snapshot_data'].replace('\n', '')))

            os.mkdir('work/io')
            os.mkdir('work/root')

        else:
            raise RuntimeErrorWithCode('E_NO_PREV_SNAPSHOT_SETUP', None)

        try:
            executor_path = self.sync_resource(int(self.ctx['executor_resource']))
        except:
            reraise('E_EXECUTOR_RESOURCE_SYNC_FAILED')

        os.mkdir('executor')
        run_process(['tar', '-C', 'executor', '-xf', executor_path])

        if '_custom_resources_parsed' not in self.ctx:
            try:
                self.ctx['_custom_resources_parsed'] \
                    = self.__parse_custom_resources_any(self.ctx['custom_resources'])
            except:
                reraise('E_MALFORMED_CUSTOM_RESOURCES')

        custom_resources = self.__custom_resources
        if custom_resources:
            try:
                symlinks, libraries_by_type = self.__sync_custom_resources(custom_resources)
            except:
                reraise('E_CUSTOM_RESOURCES_SETUP_FAILED')

            self.ctx['_created_symlinks'] = list(symlinks)
        else:
            libraries_by_type = {lib_type: [] for lib_type in self.LIBRARY_TYPES}

        packet_snapshot_file = 'work/packet.pickle'
        last_update_message_file = 'last_update_message.pickle'
        last_update_user_summary_file = 'last_update_user_summary_file'

        argv = [
            # This will ensure that python is really python and executable
            python_virtual_env_bin_directory + '/python',
            './executor/run_sandbox_packet.py',
                '--work-dir', 'work/root',
                '--io-dir',   'work/io',
                '--task-id', str(self.id),
                '--rem-server-addr', self.ctx['rem_server_addr'],
                '--result-snapshot-file', packet_snapshot_file,
                '--last-update-message-file', last_update_message_file,
                '--last-update-user-summary-file', last_update_user_summary_file,
        ]

        # if prev_packet_snapshot_file:
        argv.extend(['--snapshot-file', prev_packet_snapshot_file])

        resume_params = self.ctx.get('resume_params')
        if resume_params:
            argv.extend(['--resume-params', json.dumps(json.loads(resume_params))])

        vaults_setup = self.ctx.get(self.VaultsSetup.name)
        if vaults_setup:
            vaults_setup = json.loads(vaults_setup)
            vaults_dir = 'vaults'
            os.mkdir(vaults_dir)
            os.chmod(vaults_dir, 0700)

            vaults_config_file = 'vaults_config.pickle'

            vaults_config = self._produce_vaults_config(vaults_setup, os.path.abspath(vaults_dir))

            with open(vaults_config_file, 'w') as out:
                json.dump(vaults_config, out, indent=3)
                #pickle.dump(vaults_config, out)

            os.chmod(vaults_config_file, 0600)

            argv.extend(['--vaults-config-file', vaults_config_file])

        env = os.environ.copy()

        env['PATH'] = ':'.join(
            filter(None,
                [python_virtual_env_bin_directory] \
                + libraries_by_type['bin'] \
                + [os.path.abspath('binlib')] \
                + [env.get('PATH', None)]
            )
        )

        env['PYTHONPATH'] = ':'.join(
            filter(None,
                [
                    # '/skynet', # FIXME env.get('PYTHONPATH', ''),
                    os.path.abspath('pylib'),
                ] \
                + libraries_by_type['py'] \
                + [
                    os.path.abspath('executor') + '/client',
                ]
            )
        )

        env['LD_LIBRARY_PATH'] = ':'.join(
            filter(None,
                [os.path.abspath('ldlib')] \
                + libraries_by_type['ld']
            )
        )

        env['REM_PACKET_SANDBOX_TASK'] = 'yes'
        env['MR_MAPREDUCELIB_USE_CURRENT_PYTHON_BINARY'] = 'yes'
        env['YT_CLIENT_USE_CURRENT_PYTHON_BINARY'] = 'yes'  # XXX Temporary name until YT-5508
        env['LOCAL_SMTP_SERVER_HOST'] = 'outbound-relay.yandex.net'
        env['TASK_TOKEN'] = self.agentr.token

        with open('environment', 'w') as out:
            for k, v in env.items():
                print >>out, '%s=%s' % (k, v)

        try:
            run_process(['sh', '-c', '/sbin/ifconfig > interfaces'])
        except:
            logging.exception("Failed to dump interfaces")

        run_process(argv, environment=env, log_prefix='executor')

        if '_created_symlinks' in self.ctx:
            for name in symlinks:
                try:
                    os.unlink(name)
                except:
                    pass

        # TODO XXX Checks (at least for snapshot_file existence)

        # TODO ttl
        # FIXME What to do on if archive is too big?

        pck_id = self.ctx.get('pck_id', 'pck-FAKE1') or 'pck-FAKE2'

        snapshot_filename = 'work-%s-%s.tar' % (pck_id, self.id)
        run_process(['tar', '-C', 'work', '-cf', snapshot_filename, './'])
        self.create_resource(
            '',
            snapshot_filename,
            rt.REM_JOBPACKET_EXECUTION_SNAPSHOT)

        stderr_files = [file for file in os.listdir('work/io/') if file.startswith('err-')]
        if stderr_files:
            stderrs_archive_filename = 'stderr-%s-%s.tar' % (pck_id, self.id)
            run_process(['tar', '-C', 'work/io', '-cf', stderrs_archive_filename] + stderr_files)
            self.create_resource('', stderrs_archive_filename, rt.REM_JOBPACKET_STDERRS)

        self.create_resource(
            '',
            last_update_message_file,
            rt.REM_JOBPACKET_GRAPH_UPDATE)

        if os.path.exists(last_update_user_summary_file):
            self.ctx['_last_update_summary'] = open(last_update_user_summary_file).read()

        # rt.REM_JOBPACKET_EXECUTION_RESULTS

    def on_execute(self):
        self.__run(self.__do_on_execute)

    def __run(self, f):
        try:
            return f()
        except RuntimeErrorWithCode:
            t, e, tb = sys.exc_info()
            code, orig_error = e.args
            self.ctx['__last_rem_error'] = (code, str(orig_error))
            if isinstance(orig_error, Exception):
                raise type(orig_error), orig_error, tb
            else:
                raise t, e, tb
        except Exception as e:
            self.ctx['__last_rem_error'] = ('E_UNKNOWN', str(e))
            raise


__Task__ = RunRemJobPacket
