# -*- coding: utf-8 -*-

from sandbox.sandboxsdk import task as st
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common import apihelpers

import sandbox.common.types.misc as ctm

import os
import logging
from types import StringTypes
from traceback import format_exc


class RunScript(st.SandboxTask):
    '''
    Run arbitrary script with specified resource as argument
    '''

    type = 'RUN_SCRIPT'

    class ScriptUrl(sp.SandboxSvnUrlParameter):
        name = 'script_url'
        description = 'Script directory svn url'
        required = False

    class ScriptCmdline(sp.SandboxStringParameter):
        name = 'cmdline'
        description = 'script cmdline. Use {resource} for resource file name, or {resource_name} for named resources'
        required = True

    class Resource(sp.ResourceSelector):
        name = 'resource'
        description = 'additional resource'
        required = False
        resource_type = ''

    class NamedResources(sp.DictRepeater, sp.SandboxStringParameter):
        name = 'named_resources'
        description = 'Mapping resource_name => resource_id (or type:ABC,attr:release:stable,attr:prod:1)'
        required = False
        resource_type = ''

    class EnvVars(sp.DictRepeater, sp.SandboxStringParameter):
        name = 'env_vars'
        description = 'Environment variables'
        required = False

    class VaultEnv(sp.SandboxStringParameter):
        name = 'vault_env'
        description = 'space-separated list of ENVVAR=vault_owner:vault_key'
        required = False

    class Retries(sp.SandboxIntegerParameter):
        name = 'retries'
        description = 'Retry N times in case of task execution error'
        required = False

    class RetryInterval(sp.SandboxIntegerParameter):
        name = 'retry_interval'
        description = 'Interval between retries (secs)'
        required = True
        default_value = 3600

    class UseArcadiaCache(sp.SandboxBoolParameter):
        '''
        If true path to the cache of Arcadia trunk will be in ARCADIA_CACHE_PATH env var
        '''
        name = 'use_arcadia_cache'
        description = 'Should use cache of Arcadia trunk on Sandbox client'
        required = False

    class SaveAsResource(sp.DictRepeater, sp.SandboxStringParameter):
        name = 'save_as_resource'
        description = 'Mapping local_path_to_result => resource_type (or type:ABC,attr:release:stable,attr:prod:1). Save these results as resources'
        required = False

    class ResourcesTTL(sp.SandboxIntegerParameter):
        name = 'resources_ttl'
        description = 'Resources TTL'
        required = False
        default_value = 4

    class Container(sp.Container):
        description = 'LXC container'
        required = False
        default_value = None

    input_parameters = [ScriptUrl,
                        ScriptCmdline,
                        Resource,
                        NamedResources,
                        EnvVars,
                        VaultEnv,
                        Retries,
                        RetryInterval,
                        UseArcadiaCache,
                        SaveAsResource,
                        ResourcesTTL,
                        Container]

    environment = (environments.SvnEnvironment(),)
    dns = ctm.DnsType.DNS64

    def get_cmdline(self, scriptdir):
        url = self.ctx.get(self.ScriptUrl.name)
        if url:
            process.run_process(['svn', 'co', url, scriptdir], log_prefix='checkout')

        def get_resource_path(rid):
            if rid is None:
                return
            if isinstance(rid, StringTypes) and rid.startswith('type'):
                type_ = None
                attrs = {}
                for item in rid.split(','):
                    value = item.split(':')
                    if value[0] == "type":
                        type_ = value[1]
                    if value[0] == "attr":
                        attrs[value[1]] = value[2]
                return self.sync_resource(apihelpers.get_last_resource_with_attrs(
                    type_,
                    attrs,
                    all_attrs=True
                ))
            return self.sync_resource(int(rid))

        resmap = dict(self.ctx.get(self.NamedResources.name) or {})
        aux_resid = self.ctx.get(self.Resource.name)
        if aux_resid:
            resmap['resource'] = aux_resid

        resources = dict((name, get_resource_path(rid)) for (name, rid) in resmap.iteritems())

        cmdline = self.ctx[self.ScriptCmdline.name].format(**resources)
        return cmdline

    def parse_process_stdout(self, proc): # To override in inherited class
        return

    def on_execute(self):
        retries_rest = '_retries_rest'
        if retries_rest not in self.ctx:
            self.ctx[retries_rest] = (self.ctx.get(self.Retries.name) or 0)

        try:
            scriptdir = self.path('script')
            os.makedirs(scriptdir)

            cmdline = self.get_cmdline(scriptdir)

            env = os.environ.copy()
            env.update(self.ctx.get(self.EnvVars.name) or {})

            use_arcadia_cache = self.ctx.get(self.UseArcadiaCache.name)
            if use_arcadia_cache:
                revision = svn.Arcadia.parse_url(self.ctx.get(self.ScriptUrl.name)).revision
                env["ARCADIA_CACHE_PATH"] = svn.Arcadia.get_arcadia_src_dir(
                    "arcadia:/arc/trunk/arcadia{}".format("@%s" % revision if revision else ""))

            vault_env = self.ctx.get(self.VaultEnv.name)
            if vault_env:
                for pair in vault_env.split():
                    (name, value) = pair.split('=')
                    (owner, key) = value.split(':')
                    value = self.get_vault_data(owner, key)
                    env[name] = value

            sp = process.run_process(cmdline, shell=True, work_dir=scriptdir, environment=env, log_prefix='script_run', wait=False)
            logs_rid = getattr(getattr(channel.task, '_log_resource', None), 'id', None)
            if logs_rid is not None:
                res = channel.rest.get_resource(logs_rid)
                if res:
                    url = '/'.join([res.proxy_url, sp.stdout_path_filename])
                    self.set_info('Process started. <a href="{0}" target="_blank">output</a>'.format(url), do_escape=False)
            sp.wait()
            process.check_process_return_code(sp)
            self.parse_process_stdout(sp)

            def parse_resource_info(resource_info):
                if resource_info is None:
                    raise RuntimeError("resource_info is None")
                if resource_info.startswith('type'):
                    type_ = None
                    attrs = {}
                    for item in resource_info.split(','):
                        value = item.split(':')
                        if value[0] == "type":
                            type_ = value[1]
                        if value[0] == "attr":
                            attrs[value[1]] = value[2]
                    return type_, attrs
                else:
                    return resource_info, {}

            save_as_resource = self.ctx.get(self.SaveAsResource.name)
            if save_as_resource:
                resources_ttl = self.ctx.get(self.ResourcesTTL.name)
                for local_path, resource_info in save_as_resource.iteritems():
                    resource_path = os.path.join(scriptdir, local_path)
                    if os.path.exists(resource_path):
                        resource_type, resource_attrs = parse_resource_info(resource_info)
                        if resources_ttl:
                            resource_attrs['ttl'] = resources_ttl
                        resource = self.create_resource(local_path, resource_path, resource_type, attributes=resource_attrs)
                        self.mark_resource_ready(resource)
        except:
            if self.ctx.get(retries_rest):
                logging.error('Task failed with error: %s' % format_exc())
                tts = self.ctx[self.RetryInterval.name]
                logging.info('Will retry in %s seconds' % tts)
                self.ctx[retries_rest] -= 1
                self.wait_time(tts)
            else:
                raise


__Task__ = RunScript
