import yaml
import os
import time

import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.projects import resource_types
import sandbox.sandboxsdk.svn as sdk_svn


from sandbox.sandboxsdk.channel import channel
import logging


def get_latest_resource(*args, **kwargs):
    resources = channel.sandbox.list_resources(order_by="-id", limit=1, status="READY", *args, **kwargs)
    if resources:
        return resources[0]
    logging.warning("Can't find latest resource: %s", kwargs)


class RunSpynetYtrank(sdk_task.SandboxTask):
    cores = 1

    class Program(sdk_parameters.ResourceSelector):
        name = 'program_binary'
        description = 'program binary'
        resource_type = resource_types.ARCADIA_PROJECT
        required = True

    class ConfPath(sdk_parameters.SandboxStringParameter):
        name = 'config_path'
        description = 'config path'

    class Token(sdk_parameters.SandboxStringParameter):
        name = "yt_token"
        description = "Yt token name (in sandbox vault)"
        required = True

    class TokenOwner(sdk_parameters.SandboxStringParameter):
        name = "token_owner"
        description = "Yt token owner (in sandbox vault)"
        required = True

    class CmdTemplate(sdk_parameters.SandboxStringParameter):
        name = 'cmd_template'
        default_value = '{binary} --conf {conf} --spynet_conf {spynet_conf}'
        required = True
        description = 'Command line template, use {binary}, {conf}, {spynet_conf} and {resource_name} variables'

    class NamedResources(sdk_parameters.DictRepeater, sdk_parameters.SandboxStringParameter):
        name = 'named_resources'
        description = 'Mapping resource_name => resource_id (can be used in command line template)'

    class Environ(sdk_parameters.DictRepeater, sdk_parameters.SandboxStringParameter):
        name = 'environ'
        description = 'Environment of process'

    class MapreduceYt(sdk_parameters.ResourceSelector):
        name = 'mapreduce_yt'
        description = 'mapreduce-yt binary'
        resource_type = resource_types.ARCADIA_PROJECT
        required = True
        default = 170077745

    class ConfPatch(sdk_parameters.SandboxStringParameter):
        name = 'conf_patch'
        description = 'Patch to yabs.conf'
        multiline = True

    class SpynetConf(sdk_parameters.SandboxStringParameter):
        name = 'spynet_conf'
        description = 'Patch to spynet config'
        multiline = True

    type = "RUN_SPYNET_YTRANK"
    input_parameters = [
        Program, ConfPath, Token, TokenOwner, NamedResources,
        CmdTemplate, Environ, MapreduceYt, ConfPatch, SpynetConf
    ]

    def on_execute(self):
        binary = self.sync_resource(self.ctx[self.Program.name])
        mapreduce_yt = self.sync_resource(self.ctx[self.MapreduceYt.name])
        if self.ctx.get(self.ConfPath.name):
            for i in range(10):
                try:
                    conf_dir = sdk_svn.Arcadia.get_arcadia_src_dir(
                        sdk_svn.Arcadia.trunk_url(os.path.dirname(self.ctx[self.ConfPath.name]))
                    )
                    break
                except Exception:
                    if i == 9:
                        raise
                    time.sleep(10)
            conf = os.path.join(conf_dir, os.path.basename(self.ctx[self.ConfPath.name]))
        else:
            conf = None
        context = dict(binary=binary, conf=conf)
        resmap = dict(self.ctx.get(self.NamedResources.name) or {})
        resources = dict((name, self.sync_resource(int(rid))) for (name, rid) in resmap.iteritems())
        context.update(resources)
        env = (self.ctx[self.Environ.name] or {}).copy()
        conf_patches = os.path.join(os.path.abspath(os.getcwd()), 'patch_conf')
        if not os.path.isdir(conf_patches):
            os.mkdir(conf_patches)
            with open(os.path.join(conf_patches, 'mr.conf'), 'w') as out:
                yaml.dump({'mapreducelib': {'binaries': {'mapreduce-yt': [os.path.abspath(mapreduce_yt)]}}}, out)

            with open(os.path.join(conf_patches, 'scriptcontrol.conf'), 'w') as out:
                yaml.dump({'scriptcontrol2': {'enabled': False}, 'path': {'locks_dir': os.getcwd()}}, out)

            if self.ctx.get(self.ConfPatch.name):
                with open(os.path.join(conf_patches, 'task.conf'), 'w') as out:
                    out.write(self.ctx[self.ConfPatch.name])

        spynet_conf = os.path.join(os.path.abspath(os.getcwd()), "spynet.conf")
        with open(spynet_conf, "w") as f:
            f.write(self.ctx[self.SpynetConf.name])
        context["spynet_conf"] = spynet_conf

        env['YT_TOKEN'] = self.get_vault_data(self.ctx[self.TokenOwner.name], self.ctx[self.Token.name])
        env['YABS_ADDITIONAL_CONF_PATH'] = conf_patches

        sdk_process.run_process(
            self.ctx[self.CmdTemplate.name].format(**context),
            shell=True,
            wait=True,
            log_prefix='user_script',
            environment=env,
        )


__Task__ = RunSpynetYtrank
