# -*- coding: utf-8 -*-

import requests
import datetime
import time

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters


class RunYQL(task.SandboxTask):
    """
        Executes YQL query
    """
    type = 'RUN_YQL'

    cores = 1
    required_ram = 1024
    execution_space = 1024

    CTX_REDEFINES = {
        'kill_timeout': 25 * 60,
        'do_not_restart': True,
    }

    YQL_API_BASE_URL = 'https://yql.yandex.net/api/v2/'
    YQL_WEBUI_BASE_URL = 'https://yql.yandex-team.ru/'

    class Query(parameters.SandboxStringParameter):
        name = 'query'
        description = 'YQL Query'
        multiline = True
        required = True

    class PublishQuery(parameters.SandboxBoolParameter):
        name = 'publish_query'
        description = 'Publish executed query and show link in task results'
        default_value = False

    class SyntaxVersion1(parameters.SandboxBoolParameter):
        name = 'syntax_v1'
        description = 'Use SQLv1'
        default_value = True

    class CustomPlaceholders(parameters.DictRepeater, parameters.SandboxStringParameter):
        name = 'custom_placeholders'
        description = 'Custom placeholders'
        default_value = ''

        @classmethod
        def cast(cls, value):
            dict_value = super(RunYQL.CustomPlaceholders, cls).cast(value)
            if not dict_value:
                return None
            for placeholder in dict_value.iterkeys():
                if not placeholder.startswith("%") or not placeholder.endswith("%"):
                    raise ValueError('Placeholder must start and end with "%", but got "{}" instead'.format(placeholder))
            return dict_value

    class VaultKeyOwner(parameters.SandboxStringParameter):
        name = 'vault_key_owner'
        description = 'Vault key owner (by default task owner)'

    input_parameters = [Query, PublishQuery, SyntaxVersion1, CustomPlaceholders, VaultKeyOwner]

    def initCtx(self):
        self.ctx.update(self.CTX_REDEFINES)

    def on_execute(self):
        vault_key_owner = self.ctx.get(self.VaultKeyOwner.name) or self.owner
        token = self.get_vault_data(vault_key_owner, 'YQL_TOKEN')
        query = self.ctx.get(self.Query.name, '')

        now = datetime.datetime.now()
        utc_now = datetime.datetime.utcnow()
        yesterday = now - datetime.timedelta(days=1)
        two_days_ago = now - datetime.timedelta(days=2)
        week_ago = now - datetime.timedelta(days=7)
        two_weeks_ago = now - datetime.timedelta(days=14)
        yql_query_placeholders = {
            '%YEAR%': now.year,
            '%MONTH%': '%02d' % now.month,
            '%DAY%': '%02d' % now.day,
            '%HOUR%': '%02d' % now.hour,
            '%MINUTE%': '%02d' % now.minute,
            '%SECOND%': '%02d' % now.second,
            '%UTC_YEAR%': utc_now.year,
            '%UTC_MONTH%': '%02d' % utc_now.month,
            '%UTC_DAY%': '%02d' % utc_now.day,
            '%UTC_HOUR%': '%02d' % utc_now.hour,
            '%UTC_MINUTE%': '%02d' % utc_now.minute,
            '%UTC_SECOND%': '%02d' % utc_now.second,
            '%YESTERDAY_YEAR%': yesterday.year,
            '%YESTERDAY_MONTH%': '%02d' % yesterday.month,
            '%YESTERDAY_DAY%': '%02d' % yesterday.day,
            '%TWO_DAYS_AGO_YEAR%': two_days_ago.year,
            '%TWO_DAYS_AGO_MONTH%': '%02d' % two_days_ago.month,
            '%TWO_DAYS_AGO_DAY%': '%02d' % two_days_ago.day,
            '%WEEK_AGO_YEAR%': week_ago.year,
            '%WEEK_AGO_MONTH%': '%02d' % week_ago.month,
            '%WEEK_AGO_DAY%': '%02d' % week_ago.day,
            '%TWO_WEEKS_AGO_YEAR%': two_weeks_ago.year,
            '%TWO_WEEKS_AGO_MONTH%': '%02d' % two_weeks_ago.month,
            '%TWO_WEEKS_AGO_DAY%': '%02d' % two_weeks_ago.day,
            '%TIMESTAMP%': int(time.mktime(now.timetuple())),
            '%USER%': self.owner,
            '%OWNER%': self.owner
        }

        custom_placeholders = self.ctx.get(self.CustomPlaceholders.name)
        if custom_placeholders:
            yql_query_placeholders.update(custom_placeholders)

        for key, value in yql_query_placeholders.iteritems():
            query = query.replace(key, str(value))

        session = requests.Session()
        session.headers.update({
            'User-Agent': 'YQL Sandbox (RUN_YQL task)',
            'Authorization': "OAuth {}".format(token),
            'Content-Type': 'application/json',
        })

        request = {
            'content': query,
            'action': 'RUN',
            'type': 'SQLv1' if self.ctx.get(self.SyntaxVersion1.name) else 'SQL'
        }
        result = session.post(
            self.YQL_API_BASE_URL + 'operations',
            json=request
        )

        if result.status_code != 200:
            raise errors.SandboxTaskFailureError(
                'YQL API Error ({}):\n{}'.format(result.status_code, result.content)
            )

        self._add_link('Operation', result.json()['id'])

        if self.ctx.get(self.PublishQuery.name):
            share_id = session.get(
                "{}operations/{}/share_id".format(self.YQL_API_BASE_URL, result.json()['id'])
            )

            if share_id.status_code != 200:
                self.set_info('Error getting operation public link ({}): {}'.format(share_id.status_code, share_id.content))
            else:
                self._add_link('Public link', share_id.json())

    def _add_link(self, title, operation_id):
        url = '{}Operations/{}'.format(self.YQL_WEBUI_BASE_URL, operation_id)
        self.set_info('{0}: <a href="{1}">{1}</a>'.format(title, url), do_escape=False)


__Task__ = RunYQL
