import yaml
import os
import time

from sandbox import sdk2
import sandbox.sandboxsdk.task as sdk_task
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.projects import resource_types
import sandbox.sandboxsdk.svn as sdk_svn

from sandbox.sandboxsdk.channel import channel
import logging


def get_latest_resource(*args, **kwargs):
    resources = channel.sandbox.list_resources(order_by="-id", limit=1, status="READY", *args, **kwargs)
    if resources:
        return resources[0]
    logging.warning("Can't find latest resource: %s", kwargs)


class RunYabsPythonScript(sdk_task.SandboxTask):
    """Clone and run nirvana graph, changing some options"""
    cores = 1

    class Program(sdk_parameters.ResourceSelector):
        name = 'program_binary'
        description = 'program binary'
        resource_type = resource_types.ARCADIA_PROJECT
        required = True

    class ConfPath(sdk_parameters.SandboxStringParameter):
        name = 'config_path'
        description = 'config path'

    class Token(sdk_parameters.SandboxStringParameter):
        name = "yt_token"
        description = "Yt token name (in sandbox vault)"
        required = True

    class PushClientToken(sdk_parameters.SandboxStringParameter):
        name = "push_client_token"
        description = "TVM token name for push client (in sandbox vault)"
        required = False

    class TokenOwner(sdk_parameters.SandboxStringParameter):
        name = "token_owner"
        description = "Token owner (in sandbox vault)"
        required = True

    class CmdTemplate(sdk_parameters.SandboxStringParameter):
        name = 'cmd_template'
        default_value = '{binary} --conf {conf}'
        required = True
        description = 'Command line template, use {binary}, {conf} and {resource_name} variables'

    class NamedResources(sdk_parameters.DictRepeater, sdk_parameters.SandboxStringParameter):
        name = 'named_resources'
        description = 'Mapping resource_name => resource_id (can be used in command line template)'

    class Environ(sdk_parameters.DictRepeater, sdk_parameters.SandboxStringParameter):
        name = 'environ'
        description = 'Environment of process'

    class MapreduceYt(sdk_parameters.ResourceSelector):
        name = 'mapreduce_yt'
        description = 'mapreduce-yt binary'
        resource_type = resource_types.ARCADIA_PROJECT
        required = True
        default = 170077745

    class FormatConf(sdk_parameters.SandboxBoolParameter):
        name = 'format_conf'
        description = 'Format config string to replace resources (use {{ to write a {)'
        default_value = False

    class ConfPatch(sdk_parameters.SandboxStringParameter):
        name = 'conf_patch'
        description = 'Patch to yabs.conf'
        multiline = True

    type = "RUN_YABS_PYTHON_SCRIPT"
    input_parameters = [
        Program, ConfPath, Token, PushClientToken, TokenOwner, NamedResources, CmdTemplate, Environ,
        MapreduceYt, FormatConf, ConfPatch
    ]

    def on_execute(self):
        binary = self.sync_resource(self.ctx[self.Program.name])
        mapreduce_yt = self.sync_resource(self.ctx[self.MapreduceYt.name])
        resmap = dict(self.ctx.get(self.NamedResources.name) or {})
        resources = dict((name, self.sync_resource(int(rid))) for (name, rid) in resmap.iteritems())
        if self.ctx.get(self.ConfPath.name):
            for i in range(10):
                try:
                    conf_dir = sdk_svn.Arcadia.checkout(
                        sdk_svn.Arcadia.trunk_url(os.path.dirname(self.ctx[self.ConfPath.name])),
                        "conf_dir"
                    )
                    break
                except Exception:
                    if i == 9:
                        raise
                    time.sleep(10)
            if self.ctx.get(self.FormatConf.name):
                conf = os.path.join(os.path.abspath(os.getcwd()), "ads.sandbox_task.conf~")
                with open(os.path.join(conf_dir, os.path.basename(self.ctx[self.ConfPath.name]))) as f:
                    with open(conf, "w") as f2:
                        content = f.read().format(**resources)
                        logging.warning(content)
                        f2.write(content)
            else:
                conf = os.path.join(conf_dir, os.path.basename(self.ctx[self.ConfPath.name]))
        else:
            conf = None
        context = dict(binary=binary, conf=conf)
        context.update(resources)
        env = (self.ctx[self.Environ.name] or {}).copy()
        for env_key in env.keys():
            if env[env_key] and env[env_key].startswith('$(vault:value:'):
                (_, _, owner, name) = env[env_key].strip().split(':')
                name = name[:-1]  # cut trailing ')'
                env[env_key] = sdk2.Vault.data(owner, name)

        conf_patches = os.path.join(os.path.abspath(os.getcwd()), 'patch_conf')
        if not os.path.isdir(conf_patches):
            os.mkdir(conf_patches)
            with open(os.path.join(conf_patches, 'mr.conf'), 'w') as out:
                yaml.dump({'mapreducelib': {'binaries': {'mapreduce-yt': [os.path.abspath(mapreduce_yt)]}}}, out)

            with open(os.path.join(conf_patches, 'scriptcontrol.conf'), 'w') as out:
                yaml.dump({'scriptcontrol2': {'enabled': False}, 'path': {'locks_dir': os.getcwd()}}, out)

            if self.ctx.get(self.ConfPatch.name):
                with open(os.path.join(conf_patches, 'task.conf'), 'w') as out:
                    conf = self.ctx[self.ConfPatch.name]
                    if self.ctx.get(self.FormatConf.name):
                        conf = conf.format(**resources)
                    out.write(conf)

        env['YT_TOKEN'] = self.get_vault_data(self.ctx[self.TokenOwner.name], self.ctx[self.Token.name])
        if self.ctx.get(self.PushClientToken.name):
            env['PUSH_CLIENT_TVM_SECRET'] = self.get_vault_data(self.ctx[self.TokenOwner.name], self.ctx[self.PushClientToken.name])
        env['YABS_ADDITIONAL_CONF_PATH'] = conf_patches
        env['YQL_TOKEN'] = env['YT_TOKEN']
        sdk_process.run_process(
            self.ctx[self.CmdTemplate.name].format(**context),
            shell=True,
            wait=True,
            log_prefix='user_script',
            environment=env,
        )


__Task__ = RunYabsPythonScript
