# -*- coding: utf-8 -*-
from __future__ import absolute_import, unicode_literals
import logging

try:
    from yt_worker import YtWorker, yt
except ImportError:
    from .yt_worker import YtWorker, yt


logger = logging.getLogger(__name__)


class YtFailedCommandsAnalysis(YtWorker):
    def __init__(self, token):

        super(YtFailedCommandsAnalysis, self).__init__(token=token)

        # Create required folder
        self.yt.create("map_node", self.analysis_fails_stats,
                       recursive=True, ignore_existing=True)
        self.threshold = 20

        # Store paths to all result tables in list
        self.dest_tables_paths = []

    # Map functions
    @yt.aggregator
    def _map_users_failed_commands(self, records):
        count_per_cluster = dict()
        for record in records:
            # Prepare required fields
            user = record.get("user")
            cluster = record.get("cluster")
            done_status = record.get("done_status")

            if done_status == "False" and user is not None and cluster is not None:
                if user not in count_per_cluster:
                    count_per_cluster[user] = {cluster: 1}
                else:
                    if cluster in count_per_cluster[user]:
                        count_per_cluster[user][cluster] += 1
                    else:
                        count_per_cluster[user][cluster] = 1

        for user, value in count_per_cluster.items():
            mapped_record = dict()
            mapped_record["user"] = user
            for cluster, count in value.items():
                mapped_record["cluster"] = cluster
                mapped_record["count"] = count
                if count > self.threshold:
                    yield mapped_record

    # Reduce functions
    @staticmethod
    def _reduce_users_failed_commands(key, records):
        user, cluster = key["user"], key["cluster"]
        res_count = 0
        reduced_record = dict()
        reduced_record["user"] = user
        reduced_record["cluster"] = cluster
        for record in records:
            count = record.get("count")
            res_count += count

        reduced_record["count"] = res_count
        yield reduced_record

    # Main functions
    def run_analysis(self):
        if self.yt.exists(self.enrich_table):
            dest_table = self.analysis_fails_stats + "/users_failed_commands"
            self.dest_tables_paths.append(dest_table)
            # Map -> Sort -> Reduce
            self.yt.run_map(self._map_users_failed_commands,
                            self.enrich_table, dest_table)
            self.yt.run_sort(dest_table, sort_by=["user", "cluster"])
            self.yt.run_reduce(self._reduce_users_failed_commands, dest_table, dest_table,
                               reduce_by=["user", "cluster"])
            self.yt.run_sort(dest_table, sort_by="count")
            return True
        else:
            logger.error("Enriched table not exist!")
            return False


def main():

    analyze = YtFailedCommandsAnalysis(token=None)
    analyze.run_analysis()


if __name__ == "__main__":
    main()
