# -*- coding: utf-8 -*-
from __future__ import absolute_import, unicode_literals

try:
    from yt_raw_master import YtRawMasterAnalysis
except ImportError:
    from .yt_raw_master import YtRawMasterAnalysis

import yt.wrapper as yt
import logging

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


def prepare_secrets():
    import os
    return os.environ["YT_TOKEN"], os.environ["HEC_TOKEN"]


# Read tables from paths(list) and return dict with key equal to filename and value equal to 20 rows
def table_reader(token, paths, rows_count=None, cluster="hahn", table_format=yt.YsonFormat()):
    yt.config["token"] = token
    proxy = cluster + ".yt.yandex.net"
    yt.config["proxy"]["url"] = proxy
    result = dict()
    if rows_count is None:
        rows_count = 30

    for path in paths:
        filename = path.split("/")[-1]
        table = yt.read_table(path, table_format)
        table = list(table)
        table = table[-rows_count:]
        result[filename] = table
    return result


def send_to_splunk(hec_token, result_tables):
    from hec_sender import SplunkHECSender

    for filename, data in result_tables.items():
        source = filename
        logger.info("Calling splunk sender with source %s ...", source)
        sender = SplunkHECSender(token=hec_token, source=source)
        sender.send_data(data)


def main():
    logger.info("Preparing secrets ...")
    yt_token, hec_token = prepare_secrets()

    # Prepare objects
    raw_master = YtRawMasterAnalysis(token=yt_token)
    logger.info("starting raw-master log analysis ...")
    result = raw_master.run_user_auth_failed_analysis()

    if result:
        logger.info("Result tables found: %s", result)
        logger.info("Reading result tables ...")
        result_tables = table_reader(yt_token, raw_master.dest_paths)
        logger.info("Sending results to Splunk ...")
        send_to_splunk(hec_token, result_tables)
    else:
        logger.warning("Result tables list is empty, source table not exists!")


if __name__ == "__main__":
    main()
