# -*- coding: utf-8 -*-
from __future__ import absolute_import, unicode_literals

import yt.wrapper as yt

try:
    from acl_diff import YtAclChecker
except ImportError:
    from .acl_diff import YtAclChecker

import logging

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


def prepare_secrets():
    import os
    return os.environ["YT_TOKEN"], os.environ["HEC_TOKEN"]


# Read tables from paths(list) and return dict with key equal to filename and
# value containing rows
def table_reader(token, paths, cluster="hahn", table_format=yt.YsonFormat()):
    yt.config["token"] = token
    proxy = cluster + ".yt.yandex.net"
    yt.config["proxy"]["url"] = proxy
    result = dict()

    for path in paths:
        filename = path.split("/")[-1]
        table = yt.read_table(path, table_format)
        table = list(table)
        if len(table) > 0:
            result[filename] = table
        else:
            continue
    return result


def send_to_splunk(hec_token, result_tables):
    from hec_sender import SplunkHECSender

    for filename, data in result_tables.items():
        source = filename
        logger.info("Calling splunk sender with source %s ...", source)
        sender = SplunkHECSender(token=hec_token, source=source, sourcetype="yt-acl")
        sender.send_data(data)


def main():
    # Prepare secrets
    logger.info("Preparing secrets ...")
    yt_token, hec_token = prepare_secrets()

    clusters_for_analysis = ["banach", "hahn", "freud"]
    # Prepare objects
    acl_checker = YtAclChecker(yt_token=yt_token, hec_token=hec_token, analyze_clusters=clusters_for_analysis)
    logger.info("Running tables ACL check ...")
    acl_checker.run_check()

    # Prepare result
    logger.info("Read result table ...")
    result = table_reader(yt_token, acl_checker.results_paths)

    # Send result
    logger.info("Send result to splunk ...")
    send_to_splunk(hec_token, result)


if __name__ == "__main__":
    main()
