# -*- coding: utf-8 -*-
from __future__ import absolute_import, unicode_literals
import yt.wrapper as yt
from datetime import datetime, date, timedelta
import logging

logger = logging.getLogger(__name__)


# This is base Class
class YtWorker(object):
    """
    Parent class for main. Contains most of settings and secondary functions.
    Define here Yt config parameters and default paths.
    """

    def __init__(self, token, analyze_clusters=None, cluster="hahn", source_path="//home/acl_dump",
                 dest_path="//home/infrasec/yt-acl"):
        # Config yt parameters
        self.yt = yt
        self.cluster = cluster
        self.proxy = cluster + ".yt.yandex.net"
        self.yt_token = token
        self.yt.config["token"] = self.yt_token
        self.yt.config["proxy"]["url"] = self.proxy
        self.yt.config["auto_merge_output"]["action"] = "merge"

        # Filters for compability
        self.yt.config['pickling']['module_filter'] = lambda module: hasattr(module, '__file__') and \
            not module.__file__.endswith('.so') and \
            'hashlib' not in getattr(module, '__name__', '') and \
            'hmac' not in getattr(module, '__name__', '')

        self.yt.config['pickling']['force_using_py_instead_of_pyc'] = True
        # Tunning maximum resource usage
        self.yt_spec = {"job_io": {"table_writer": {"max_row_weight": 128 * 1024 * 1024}}}

        self.current_datetime = datetime.now()
        self.ttl = (self.current_datetime + timedelta(days=7)).isoformat() + "+03:00"

        self.tmp_ttl = (self.current_datetime + timedelta(hours=5)).isoformat() + "+03:00"
        # Clusters list
        if analyze_clusters is not None and type(analyze_clusters) is list:
            self.clusters = analyze_clusters
        elif analyze_clusters is None:
            self.clusters = ["banach", "freud", "hahn", "hume"]
        elif type(analyze_clusters) is bool and analyze_clusters:
            # Assign all clusters
            self.clusters = ["banach", "hahn", "freud",
                             "flux", "hume", "locke",
                             "markov", "perelman", "pythia",
                             "seneca-man", "seneca-myt", "seneca-sas",
                             "vanga", "zeno"]

        # Define paths: source -- path YT logs in logbroker, dest -- path to our account storage
        self.source_path = source_path
        self.dest_path = dest_path

    # Get date methods
    @staticmethod
    def get_current_date(delim=False):
        if not delim:
            today = date.today().strftime("%Y%m%d")
        else:
            today = date.today().strftime("%Y-%m-%d")
        return today

    @staticmethod
    def get_yesterday_date(base_date=None, delim=None):
        if delim is None and base_date is None:
            yesterday = (date.today() - timedelta(1)).strftime("%Y%m%d")
        elif delim is not None and base_date is None:
            timeformat = ["%Y", "%m", "%d"]
            yesterday = (date.today() - timedelta(1)).strftime(delim.join(timeformat))
        elif delim is None and base_date is not None:
            try:
                yesterday = (datetime.strptime(base_date, "%Y%m%d") - timedelta(1)).strftime("%Y%m%d")
            except ValueError as err:
                logger.error("Wrong datetime format! Using current date instead.", err)
                yesterday = (date.today() - timedelta(1)).strftime("%Y%m%d")
        elif delim is not None and base_date is not None:
            timeformat = ["%Y", "%m", "%d"]
            try:
                yesterday = (datetime.strptime(base_date, delim.join(timeformat)) -
                             timedelta(1)).strftime(delim.join(timeformat))
            except ValueError as err:
                logger.error("Wrong datetime format! Using current date instead.", err)
                yesterday = (date.today() - timedelta(1)).strftime("%Y-%m-%d")

        return yesterday

    @staticmethod
    def check_timeformat(timestring):
        try:
            datetime.strptime(timestring, "%Y%m%d")
            return True
        except ValueError:
            raise ValueError("Wrong datetime format! Time string must be like: %Y%m%d")
