# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.task import SandboxTask

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import copy_path, make_folder
from sandbox.sandboxsdk.parameters import ResourceSelector, ListRepeater, SandboxStringParameter, SandboxRadioParameter, SandboxBoolParameter
from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers
from sandbox.projects.SaasRoadsGraphTrafficHistoryBuilder import GraphResourceTypeParameter, GraphResourceIdParameter, ImportTrackThresholdParameter, ImportSpeedThresholdParameter, SaasRoadsGraphTrafficHistoryBuilder, YtWorkDirParameter
from sandbox.projects.TestRTYServerTask.resources import ResourceManager
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny import const
from sandbox.projects.geobase.Geodata5BinStable import resource as gbr
from sandbox.projects.masstransit.resources import MASSTRANSIT_DATA_RASP

import logging
import os
import httplib


class PConverterId(ResourceSelector):
    resource_type = resource_types.RTYSERVER_UTILS_ROADS_GRAPH
    required = False
    name, description = 'converter_resource_id', 'roads_graph binary'


class PGardenExportId(ResourceSelector):
    resource_type = resource_types.RTYSERVER_GARDEN_EXPORT
    required = False
    name, description = 'garden_export_resource_id', 'maps garden export'


class PAdditionalGardenExportIds(ListRepeater, SandboxStringParameter):
    required = False
    name, description = 'additional_garden_export_resource_ids', 'additional saas graphs resources ids'


class PFakeConnectionsId(ResourceSelector):
    resource_type = resource_types.RTYSERVER_GRAPH_FAKE_CONNECTIONS
    required = False
    name, description = 'graph_fake_connections_resource_id', 'fake connections for graph'


class PMassTransportExportId(ResourceSelector):
    resource_type = MASSTRANSIT_DATA_RASP
    required = False
    name, description = 'masstransport_export_resource_id', 'mass transport export'


class PNeedTransportRoutes(SandboxBoolParameter):
    required = False
    name, description = 'need_transport_routes', 'Need transport routes'
    default_value = False


class PNeedTransportStops(SandboxBoolParameter):
    required = False
    name, description = 'need_transport_stops', 'Need transport stops'
    default_value = False


class PNeedTransportSubway(SandboxBoolParameter):
    required = False
    name, description = 'need_transport_subway', 'Need transport subway'
    default_value = False


class PMoscowOnly(SandboxBoolParameter):
    required = False
    name, description = 'only_moscow', 'Export only Moscow'
    default_value = False


class PGeodata5Id(ResourceSelector):
    resource_type = gbr.GEODATA5BIN_STABLE
    required = False
    name, description = 'geodata5_resource_id', 'geodata5.bin file'


class PGeoTZDataId(ResourceSelector):
    resource_type = resource_types.GEODATATZDATA_STABLE
    required = False
    name, description = 'geotzdata_resource_id', 'zones* folders'


class PSaasGraphs(ListRepeater, SandboxStringParameter):
    required = False
    name, description = 'saas_graphs_ids', 'converted saas graphs resources ids'


class PMode(SandboxRadioParameter):
    required = True
    choices = [(mode, mode) for mode in ['import', 'merge', 'traffic_history', 'fake']]
    default_value = 'import'
    name, description = 'mode', 'working mode'


class PResultResourceType(SandboxStringParameter):
    required = True
    name, description = 'result_resource_type', 'result resource type'


class PAutoRelease(ListRepeater, SandboxStringParameter):
    required = False
    choices = ['stable', 'prestable']
    name, description = 'auto_release', 'Auto release to'


class PMailForProblemReport(ListRepeater, SandboxStringParameter):
    required = False
    name, description = 'mail_for_problem_report', 'Mail for problem report'


class PAutoGenerated(SandboxBoolParameter):
    required = False
    name, description = 'auto_generated', 'Auto generated'
    default_value = False


def create_graph_name(attrs):
    return '%s_%s_%s' % (
    attrs.get('garden_environment', 'noenv'), attrs.get('region', 'noregion'), attrs.get('shipping_date', 'noshipping'))


def get_graph_name(attrs):
    if 'graph_name' in attrs:
        return attrs['graph_name']
    else:
        return create_graph_name(attrs)


class SaasGardenGraphConverter(nanny.ReleaseToNannyTask, SandboxTask):
    type = 'SAAS_GARDEN_GRAPH_CONVERT'
    INPUT_GRAPHS_IDS_ATTR = 'input_graphs_ids'
    INDEX_RESOURCE_ID_ATTR = 'index_resource_id'
    AUTO_GENERATED_ATTR = 'auto_generated'

    input_parameters = [PConverterId,
                        PGardenExportId,
                        PAdditionalGardenExportIds,
                        PMassTransportExportId,
                        PNeedTransportRoutes,
                        PNeedTransportStops,
                        PNeedTransportSubway,
                        PResultResourceType,
                        PGeodata5Id,
                        PGeoTZDataId,
                        PAutoRelease,
                        PMode,
                        PSaasGraphs,
                        PMailForProblemReport,
                        PAutoGenerated,
                        PFakeConnectionsId,
                        PMoscowOnly,
                        nanny.LabelsParameter
                        ]

    execution_space = 50000

    mails = ['nsofya', 'ivanmorozov', 'svshevtsov']

    check_metrics = [
        ['VerticesCount', 0.9, 1.1],
        ['EdgesCount', 0.9, 1.1],
        ['GeoObjectsCount', 0.9, 1.1],
        ['ManeuversCount', 0.9, 1.1],
        ['DeadEndsCount', 0, 1.1],
        ['OneDirectionDeadEndsCount', 0, 1.1],
        ['IsolatedEdgesCount', 0, 1.1],
        ['AverageSnapshotEdgesCount', 0.9, 10],
        ['AverageSnapshotTracksCount', 0.9, 10],
    ]

    def mark_released_resources(self, status, ttl="inf"):
        SandboxTask.mark_released_resources(self, status, 180)
        rest = channel.rest.server
        for resource in self.list_resources(self.ctx[PResultResourceType.name]):
            logging.info("mark %s released to %s" % (resource, status))
            try:
                rest.resource[resource.id].attribute.create(
                    name="released_to_" + status,
                    value="yes"
                )
            except rest.HTTPError as ex:
                if ex.status == httplib.CONFLICT:
                    pass
                else:
                    raise

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        mailbody = additional_parameters.get('release_comments', '') + '\nNanny releases:'
        for nanny_releases_id in self.ctx.get("nanny_release_requests", []):
            release_request_link = const.RELEASE_REQUEST_TEMPLATE.format(
                nanny_api_url=const.NANNY_API_URL,
                release_request_id=nanny_releases_id
            )
            mailbody += '\n  <a href="{}">{}</a>'.format(release_request_link, nanny_releases_id)
        channel.sandbox.send_email(self.mails, None, 'Begin deploy graph %s' % (additional_parameters.get('release_header', '')), mailbody)

    def check_auto_release_enabled(self, attrs, ctype):
        search_attrs = {}
        search_attrs["released_to_" + ctype] = "yes"
        search_attrs["region"] = attrs["region"]
        last_released = apihelpers.get_last_resource_with_attrs(self.ctx[PResultResourceType.name], attrs=search_attrs, all_attrs=True)
        result = {'ctype': ctype}
        if not last_released:
            result['resolution'] = '    Reason: First release'
            result['autorelease'] = False
        else:
            result['resolution'] = '    Last release: %s\n    Metrics:' % last_released
            result['autorelease'] = True
            for metric in self.check_metrics:
                if (metric[0] in last_released.attributes and metric[0] in attrs):
                    last_value = float(last_released.attributes[metric[0]])
                    minValue = last_value * metric[1]
                    maxValue = last_value * metric[2]
                    value = float(attrs[metric[0]])
                    status = 'Ok'
                    if value < minValue or value > maxValue:
                        result['autorelease'] = False
                        status = 'Fail'
                    result['resolution'] += '\n      %s: %s -> %s (%s)' % (metric[0], last_value, value, status)
        return result

    def auto_release(self, new_resource, attrs):
        if not self.ctx.get(PAutoRelease.name):
            return

        release_to = []
        for release_candidate in self.ctx[PAutoRelease.name]:
            release_to.append(self.check_auto_release_enabled(attrs, release_candidate))

        message_header = 'Task: %s/resources' % self.http_url() + \
                         '\nShipping date: %s' % attrs.get('shipping_date') + \
                         '\nResource: %s\n' % new_resource

        message = message_header + 'Releases:'
        not_released = []
        for release in release_to:
            message += '\n  ' + release['ctype'] + ':\n    Released:' + str(release['autorelease']) + '\n' + release['resolution']
            if not release['autorelease']:
                not_released.append(release['ctype'])

        logging.info(message)
        if not_released:
            release_status = ' Will not be released to %s.' % ','.join(not_released)
        else:
            release_status = ''

        channel.sandbox.send_email(self.mails, None,
                                   'Saas graph created for %s %s.%s' % (
                                   attrs.get('garden_environment'), attrs.get('region'), release_status),
                                   message)

        subtasks = []
        for release in release_to:
            if release['autorelease']:
                ctype = release['ctype']
                release_header = '%s %s (task %s) to %s' % (attrs.get('region'), attrs.get('shipping_date'), self.id, ctype)
                release_comments = message_header + release['resolution']
                subtasks.append(self.create_subtask(
                        task_type='RELEASE_ANY',
                        inherit_notifications=True,
                        input_parameters={
                            'wait_tasks_ids': subtasks,
                            'release_task_id': self.id,
                            'release_status': ctype,
                            'release_comments': release_comments,
                            'release_subject': 'Autorelease graph %s' % release_header,
                            'release_addresses_to': self.mails,
                            'release_additional_parameters': {'release_header': release_header,
                                                              'release_comments': release_comments}
                        },
                        description="Saas graph (task id: " + str(self.id) + ') autorelease to ' + ctype
                    ).id)

    def get_converter(self):
        converter_resource_id = self.ctx[PConverterId.name]
        if not converter_resource_id:
            converter = apihelpers.get_last_released_resource(resource_types.RTYSERVER_UTILS_ROADS_GRAPH)
            if not converter:
                raise errors.SandboxTaskFailureError("I don't know witch roads_graph to use")
            logging.info('converter selected: %s' % converter)
            converter_resource_id = converter.id

        logging.info('converter resource id %s' % converter_resource_id)
        return channel.task.sync_resource(converter_resource_id)

    def get_fake_connections(self):
        connections_resource_id = self.ctx[PFakeConnectionsId.name]
        if not connections_resource_id:
            connections = apihelpers.get_last_released_resource(PFakeConnectionsId.resource_type)
            if not connections:
                raise errors.SandboxTaskFailureError("Cannot obtain last released fakeconnections")
            logging.info('fake_connections resource  %s' % connections)
            connections_resource_id = connections.id

        return channel.task.sync_resource(connections_resource_id)

    def get_garden_export(self, id):
        garden_export_resource_id = id
        garden_export_resource = channel.sandbox.get_resource(garden_export_resource_id)
        if not garden_export_resource:
            raise errors.SandboxTaskFailureError("There is not garden export resource with id %s" % garden_export_resource_id)

        logging.info('unpacking %s' % garden_export_resource)
        tar_path = os.path.join(channel.task.sync_resource(garden_export_resource_id), 'graph_data.tar.gz')
        untar_path = make_folder('graph_import_' + str(garden_export_resource_id), True)
        ResourceManager().unpackTar(tar_path, untar_path)
        logging.info('unpacked %s' % garden_export_resource)

        return untar_path

    def get_garden_export_attributes(self, id):
        garden_export_resource_id = id
        garden_export_resource = channel.sandbox.get_resource(garden_export_resource_id)
        if not garden_export_resource:
            raise errors.SandboxTaskFailureError("There is not garden export resource with id %s" % garden_export_resource_id)

        return garden_export_resource.attributes

    def get_geodata(self):
        geodata_resource_id = self.ctx[PGeodata5Id.name]
        if not geodata_resource_id:
            geodata = apihelpers.get_last_released_resource(PGeodata5Id.resource_type)
            if not geodata:
                raise errors.SandboxTaskFailureError("Cannot obtain last released geodata5.bin")
            logging.info('geodata5.bin selected: %s' % geodata)
            geodata_resource_id = geodata.id

        logging.info('geodata5.bin resource id %s' % geodata_resource_id)
        return channel.task.sync_resource(geodata_resource_id)

    def get_geotzdata(self):
        geotzdata_resource_id = self.ctx[PGeoTZDataId.name]
        if not geotzdata_resource_id:
            geotzdata = apihelpers.get_last_released_resource(PGeoTZDataId.resource_type)
            if not geotzdata:
                raise errors.SandboxTaskFailureError("Cannot obtain last released zones*")
            logging.info('zones* selected: %s' % geotzdata)
            geotzdata_resource_id = geotzdata.id

        logging.info('zones* resource id %s' % geotzdata_resource_id)
        return channel.task.sync_resource(geotzdata_resource_id)

    dest_path = ''

    def get_report_dir(self):
        return os.path.join(self.dest_path, 'problem_reports')

    def get_report_path(self, graph_name):
        return os.path.join(self.get_report_dir(), 'problems_%s' % graph_name)

    def get_res_path(self):
        return os.path.join(self.dest_path, 'graph_info')

    def parse_metrics(self, path):
        parsed_data = [{}]
        prefix = [parsed_data[0]]
        with open(path, 'r') as f:
            for line in f.readlines():
                if '{' in line:
                    sp = line.partition('{')
                    field = prefix[-1].setdefault(sp[0].strip(), [])
                    field.append({})
                    prefix.append(field[-1])
                elif '}' in line:
                    prefix.pop()
                elif ':' in line:
                    sp = line.partition(':')
                    field = prefix[-1].setdefault(sp[0].strip(), [])
                    field.append(sp[2].strip())
        return parsed_data

    def dump_metrics(self, path):
        cmd = [
            self.get_converter(), 'dump_metrics', path
        ]
        run_process(cmd, "roads_graph_dump_metrics")

    def update_attrs_by_metrics(self, attrs, parsed_data, prefix=''):
        if isinstance(parsed_data, str):
            attrs[prefix] = parsed_data
        elif isinstance(parsed_data, dict):
            if prefix:
                prefix += '.'
            for k, v in parsed_data.items():
                self.update_attrs_by_metrics(attrs, v, prefix + k)
        elif isinstance(parsed_data, list):
            if len(parsed_data) == 1:
                self.update_attrs_by_metrics(attrs, parsed_data[0], prefix)
            else:
                for i in range(0, len(parsed_data)):
                    self.update_attrs_by_metrics(attrs, parsed_data[i], '%s[%s]' % (prefix, i))
        else:
            raise "unknown type of data"

    def postprocess_graph(self, graph_name, attrs):
        if self.ctx.get(PAutoGenerated.name):
            attrs[self.AUTO_GENERATED_ATTR] = 'yes'
        path_threads_info = os.path.join(self.get_report_dir(), 'routes_metrices')
        if os.path.exists(path_threads_info):
            self.update_attrs_by_metrics(attrs, self.parse_metrics(path_threads_info))
        path = os.path.join(self.get_res_path(), 'graph.metrics')
        if os.path.exists(path):
            self.update_attrs_by_metrics(attrs, self.parse_metrics(path))
            report_path = self.get_report_dir()
            if os.path.exists(report_path):
                report_res = self._create_resource('Graph problem csv report', report_path, resource_types.ERROR_LOG)
                res_url = 'https://proxy.sandbox.yandex-team.ru/%s' % report_res.id
                logging.info('report resource:%s' % res_url)
                mails = self.ctx[PMailForProblemReport.name]
                if mails:
                    with open(path, 'r') as f:
                        channel.sandbox.send_email(
                            mails, '',
                            'Saas graph problem report for %s' % graph_name,
                            'Problem report: %s\nMetrics:\n%s' % (res_url, f.read())
                        )

    def fake_run(self, attrs):
        logging.warning('fake run')
        res_path = self.get_res_path()
        make_folder(res_path, True)
        run_process(['touch', os.path.join(res_path, 'fake')])
        return 'fake'

    def import_graph(self, attrs):
        garden_export_ids = []
        garden_export_ids.append(self.ctx[PGardenExportId.name])
        additional = self.ctx[PAdditionalGardenExportIds.name]
        if not additional:
            additional = []
        for garden_export_resource_id in additional:
            if garden_export_resource_id in garden_export_ids:
                raise errors.SandboxTaskFailureError('garden export %d is specified more than once', garden_export_resource_id)
            garden_export_ids.append(garden_export_resource_id)

        untar_path = None

        garden_environment = None
        garden_regions = []
        garden_shipping_dates = []
        exports = []
        names = []

        for garden_export_resource_id in garden_export_ids:
            untar_path = self.get_garden_export(garden_export_resource_id)
            garden_export_attrs = self.get_garden_export_attributes(garden_export_resource_id)
            environment = garden_export_attrs['garden_environment']
            region = garden_export_attrs['region']
            shipping_date = garden_export_attrs['shipping_date']
            name = create_graph_name(garden_export_attrs)

            names.append(name)
            exports.append(untar_path)

            if region in garden_regions:
                raise errors.SandboxTaskFailureError('garden export region %s occures more than once', region)
            garden_regions.append(region)

            if garden_environment is None:
                garden_environment = environment
            elif garden_environment != environment:
                raise errors.SandboxTaskFailureError('multiple garden environments')

            if shipping_date not in garden_shipping_dates:
                garden_shipping_dates.append(shipping_date)

            attrs[self.INDEX_RESOURCE_ID_ATTR + '_' + region] = str(garden_export_resource_id)
            for key, value in garden_export_attrs.iteritems():
                attrs[key + '_' + region] = value

        attrs['region'] = '-'.join(sorted(garden_regions))
        attrs['shipping_date'] = '-'.join(sorted(garden_shipping_dates))
        attrs['garden_environment'] = garden_environment

        need_transport_routes = self.ctx.get(PNeedTransportRoutes.name)
        need_transport_stops = self.ctx.get(PNeedTransportStops.name)
        need_transport_subway = self.ctx.get(PNeedTransportSubway.name)

        labels = self.ctx.get(const.LABELS)
        attrs['transport_routes'] = str(need_transport_routes)
        labels['transport_routes'] = str(need_transport_routes)
        attrs['transport_stops'] = str(need_transport_stops)
        labels['transport_stops'] = str(need_transport_stops)
        attrs['transport_subway'] = str(need_transport_subway)
        labels['transport_subway'] = str(need_transport_subway)
        self.ctx[const.LABELS] = labels

        mass_transport_path = None
        if need_transport_routes or need_transport_stops or need_transport_subway:
            mass_transport_export_resource_id = self.ctx[PMassTransportExportId.name]
            if mass_transport_export_resource_id:
                mass_transport_export_resource = channel.sandbox.get_resource(mass_transport_export_resource_id)
            else:
                mass_transport_export_resource = apihelpers.get_last_resource_with_attribute(MASSTRANSIT_DATA_RASP, "garden_environment", "production")
            if not mass_transport_export_resource or not mass_transport_export_resource.is_ready():
                raise errors.SandboxTaskFailureError("Problems with mass transport data resource exporting")
            logging.info('take mass transport from %s' % mass_transport_export_resource)
            mass_transport_path = channel.task.sync_resource(mass_transport_export_resource)

        tzdata_tar_path = self.get_geotzdata()
        tzdata_untar_path = make_folder('unpacked_tzdata', True)
        ResourceManager().unpackTar(tzdata_tar_path, tzdata_untar_path)

        fake_connections_data_path = self.get_fake_connections()
        if fake_connections_data_path:
            fc_untar_path = make_folder('fake_data', True)
            ResourceManager().unpackTar(fake_connections_data_path, fc_untar_path)
            fc_geom = os.path.join(fc_untar_path, 'fake_connections_geom')
            fc_geom_point = os.path.join(fc_untar_path, 'fake_connections_geom.point')
            copy_path(fc_geom, untar_path)
            copy_path(fc_geom_point, untar_path)

        graph_name = '-'.join(names)
        make_folder(self.get_report_dir(), True)
        cmd = [
                self.get_converter(), 'import',
                '--geobase', self.get_geodata(),
                '--zones2', os.path.join(tzdata_untar_path, "zones_bin"),
                '-e', self.dest_path,
                '--build-metrics', self.dest_path,
                '-n', attrs.get('region'),
                '-p', self.get_report_path(graph_name),
                '-o', self.get_report_dir()
        ]
        for e in exports:
            cmd.append('-i')
            cmd.append(e)
        if mass_transport_path:
            cmd.append('-T')
            cmd.append(mass_transport_path)
        if self.ctx.get(PMoscowOnly.name):
            cmd.append('-m')
        if need_transport_routes:
            cmd.append('--transport-routes')
        if need_transport_stops:
            cmd.append('--transport-stops')
        if need_transport_subway:
            cmd.append('--transport-subway')

        run_process(cmd, 'roads_graph')
        return graph_name

    def merge_graphs(self, attrs):
        input_graphs_ids = self.ctx[PSaasGraphs.name]
        if len(input_graphs_ids) < 2:
            raise errors.SandboxTaskFailureError('Need at list 2 graphs for merge, but %s get' % len(input_graphs_ids))
        names = []
        regions = []
        shipping_date = ''
        for input_id in input_graphs_ids:
            input_resource = channel.sandbox.get_resource(input_id)
            if not input_resource:
                raise errors.SandboxTaskFailureError("There is not input resource with id %s" % input_id)
            self.set_info(
                'Graph get for merge: <a href="%s">%s</a>' % (input_resource.url, input_resource),
                do_escape=False
            )
            names.append(create_graph_name(input_resource.attributes))
            regions.append(input_resource.attributes.get('region', 'noregion'))
            sd = input_resource.attributes.get('shipping_date', '')
            if sd != '' and (shipping_date == '' or shipping_date < sd):
                shipping_date = sd

        graph_name = '-'.join(sorted(names))
        make_folder(self.get_report_dir(), True)
        cmd = [self.get_converter(), 'merge',
               '-o', self.dest_path,
               '-x', '1000',
               '-y', '1000',
               '-p', self.get_report_path(graph_name)
               ]
        for input_id in input_graphs_ids:
            cmd.append(os.path.dirname(channel.task.sync_resource(input_id)))
        attrs[self.INPUT_GRAPHS_IDS_ATTR] = str(sorted(input_graphs_ids))
        attrs['region'] = ', '.join(sorted(regions))
        attrs['shipping_date'] = shipping_date
        run_process(cmd, 'roads_graph')
        return graph_name

    def traffic_history_graph(self, attrs):
        source_graph_ids = self.ctx[PSaasGraphs.name]
        if len(source_graph_ids) != 1:
            raise errors.SandboxTaskFailureError('Expected exactly 1 graph as a source for traffic history, got %s' % len(source_graph_ids))

        source_graph_id = source_graph_ids[0]
        source_graph = channel.sandbox.get_resource(source_graph_id)

        with self.memoize_stage.init:
            subtask_parameters = {
                'kill_timeout': self.ctx['kill_timeout'],
                GraphResourceIdParameter.name: source_graph.id,
                GraphResourceTypeParameter.name: self.ctx[PResultResourceType.name],
                ImportTrackThresholdParameter.name: 10,
                ImportSpeedThresholdParameter.name: 1,
                YtWorkDirParameter.name: '//tmp/extdata/rtline/traffic-history/{}'.format(source_graph.id),
            }

            subtask = self.create_subtask(task_type=SaasRoadsGraphTrafficHistoryBuilder.type,
                                          description='Traffic history for %s' % source_graph.id,
                                          input_parameters=subtask_parameters,
                                          ram=SaasRoadsGraphTrafficHistoryBuilder.RAM,
                                          execution_space=SaasRoadsGraphTrafficHistoryBuilder.EXECUTION_SPACE,
                                          )
            self.ctx['traffic_history_task_id'] = subtask.id
            self.wait_task_completed(subtask)

        for attr in ['garden_environment', 'region', 'shipping_date']:
            attrs[attr] = source_graph.attributes.get(attr, '')

        res = channel.sandbox.list_resources(resource_type=self.ctx[PResultResourceType.name],
                                             task_id=self.ctx['traffic_history_task_id'])
        assert len(res) == 1, 'Expected exactly 1 graph with traffic history to be produced, got %s' % len(res)
        graph_with_history_resource = res[0]

        need_transport_routes = self.ctx.get(PNeedTransportRoutes.name)
        need_transport_stops = self.ctx.get(PNeedTransportStops.name)
        need_transport_subway = self.ctx.get(PNeedTransportSubway.name)

        labels = self.ctx.get(const.LABELS)
        labels['transport_routes'] = str(need_transport_routes)
        labels['transport_stops'] = str(need_transport_stops)
        labels['transport_subway'] = str(need_transport_subway)
        labels['with_history'] = str(True)
        self.ctx[const.LABELS] = labels

        graph_with_history_path = channel.task.sync_resource(graph_with_history_resource.id)
        copy_path(graph_with_history_path, self.get_res_path())
        #chmod(self.dest_path, 0666)
        # self.dump_metrics(self.dest_path)
        channel.sandbox.delete_resource(graph_with_history_resource.id, ignore_last_usage_time=True)

        graph_name = create_graph_name(attrs)

        return graph_name

    def on_execute(self):
        attrs = {}
        mode = self.ctx[PMode.name]
        self.dest_path = make_folder('result_graph')
        if mode == 'import':
            graph_name = self.import_graph(attrs)
        elif mode == 'merge':
            graph_name = self.merge_graphs(attrs)
        elif mode == 'traffic_history':
            graph_name = self.traffic_history_graph(attrs)
        elif mode == 'fake':
            graph_name = self.fake_run(attrs)
        else:
            raise errors.SandboxTaskFailureError('unknown mode: %s' % mode)
        attrs['graph_name'] = graph_name
        self.postprocess_graph(graph_name, attrs)
        resource = self._create_resource('Saas graph_info', self.get_res_path(), self.ctx[PResultResourceType.name], complete=1, attrs=attrs)
        self.auto_release(resource, attrs)


__Task__ = SaasGardenGraphConverter
