# -*- coding: utf-8 -*-

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.channel import channel
from sandbox.projects.common import apihelpers
from sandbox.projects.common.nanny import const
from sandbox.projects.common.nanny import nanny
from sandbox.projects import resource_types
from sandbox.projects.SaasGardenGraphConvert import PAutoRelease, PMailForProblemReport, PMode, PGardenExportId, PAdditionalGardenExportIds, PSaasGraphs, PResultResourceType, PAutoGenerated, SaasGardenGraphConverter, get_graph_name
from sandbox.projects.SaasRoadsGraphTrafficHistoryBuilder import SaasRoadsGraphTrafficHistoryBuilder
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxStringParameter

from . import resources
import logging


class PGardenEnvironment(SandboxStringParameter):
    name, description = 'garden_environment', 'garden environment to analyze'
    default_value = 'production'


class PMailForImportProblemReport(PMailForProblemReport):
    name, description = 'mail_for_import_problem_report', 'Mail for import problem report'


class PMailForMergeProblemReport(PMailForProblemReport):
    name, description = 'mail_for_merge_problem_report', 'Mail for merge problem report'


class PNeedTransportRoutes(SandboxBoolParameter):
    required = False
    name, description = 'need_transport_routes', 'Need transport routes'
    default_value = False


class PNeedTransportStops(SandboxBoolParameter):
    required = False
    name, description = 'need_transport_stops', 'Need transport stops'
    default_value = False


class PNeedTransportSubway(SandboxBoolParameter):
    required = False
    name, description = 'need_transport_subway', 'Need transport subway'
    default_value = False


class PImportEnabled(SandboxBoolParameter):
    name, description = 'import_enabled', 'enable import graphs'
    default_value = True


class PMoscowOnly(SandboxBoolParameter):
    required = False
    name, description = 'only_moscow', 'Export only Moscow'
    default_value = False


class PMergeEnabled(SandboxBoolParameter):
    name, description = 'merge_enabled', 'enable merge graphs'
    default_value = True


class PTrafficHistoryEnabled(SandboxBoolParameter):
    name, description = 'history_enabled', 'enable traffic history'
    default_value = True


class SaasGardenHelper(nanny.ReleaseToNannyTask, SandboxTask):
    MERGE_RESOURCE_TYPE_BY_PARTS = {
        resource_types.RTYSERVER_GRAPH_DIR_PRODUCTION_RUS_AND_TURKEY: [resource_types.RTYSERVER_GRAPH_DIR_PRODUCTION_RUS, resource_types.RTYSERVER_GRAPH_DIR_PRODUCTION_TURKEY],
    }

    # Source -> result graph types for traffic history.
    TRAFFIC_HISTORY_RESOURCE_TYPES = {
        resource_types.RTYSERVER_GRAPH_DIR_PRODUCTION_RUS: resource_types.RTYSERVER_GRAPH_WITH_HISTORY_DIR_PRODUCTION_RUS,
    }

    type = 'SAAS_GARDEN_HELPER'

    input_parameters = [PGardenEnvironment, PAutoRelease, PMailForImportProblemReport, PMailForMergeProblemReport, PImportEnabled, PMergeEnabled, PTrafficHistoryEnabled, PNeedTransportSubway, PNeedTransportStops, PNeedTransportRoutes, PMoscowOnly]

    def fill_subtask_parameters(self, subtask_params):
        subtask_params[PAutoRelease.name] = self.ctx.get(PAutoRelease.name)
        subtask_params[PAutoGenerated.name] = True

    def get_not_done_subtasks(self, task):
        result = []
        for subtask in task.list_subtasks(load=True):
            if not channel.sandbox.get_task(subtask.id).is_done():
                result.append(subtask)
        return result

    def wait_subtasks_with_children(self):
        tasks = self.get_not_done_subtasks(self)
        while tasks:
            self.wait_all_tasks_stop_executing(tasks)
            new_tasks = []
            for task in tasks:
                new_tasks.extend(self.get_not_done_subtasks(task))
            tasks = new_tasks

    def build_convert_task(self, candidates, res_type):
        if self.ctx.get(PNeedTransportStops.name) or self.ctx.get(PNeedTransportRoutes.name) or self.ctx.get(PNeedTransportSubway.name):
            res_type = resources.RTYSERVER_GRAPH_ABSTRACT

        attrs = {}
        description = ''
        regions = []
        garden_environment = None
        primary_id = None
        secondary_ids = []
        for garden_export in candidates:
            if garden_export is None:
                logging.info('encountered NULL export')
                return None

            environment = garden_export.attributes['garden_environment']
            region = garden_export.attributes['region']
            attrs[SaasGardenGraphConverter.INDEX_RESOURCE_ID_ATTR + '_' + region] = str(garden_export.id)

            if garden_environment is None:
                garden_environment = environment
            elif garden_environment != environment:
                raise errors.SandboxTaskFailureError('multiple garden environments')

            if len(description) > 0:
                description += ' '
            description += (region + ':' + str(garden_export.id))

            if primary_id is None:
                primary_id = garden_export.id
            else:
                secondary_ids.append(garden_export.id)

            regions.append(region)

        logging.info('build_convert_task %s to %s' % (description, res_type))

        result_task = apihelpers.get_last_resource_with_attrs(res_type, attrs=attrs, all_attrs=True)
        if result_task is not None:
            logging.info('graph for %s already exists: %s' % (description, result_task))
            return None

        labels = {}
        labels['region'] = '-'.join(sorted(regions))
        labels['garden_environment'] = garden_environment

        subtask_parameters = {
            PGardenExportId.name: primary_id,
            PAdditionalGardenExportIds.name: secondary_ids,
            PResultResourceType.name: str(res_type),
            PMailForProblemReport.name: self.ctx.get(PMailForImportProblemReport.name),
            PMode.name: 'import',
            PNeedTransportStops.name: self.ctx.get(PNeedTransportStops.name),
            PNeedTransportRoutes.name: self.ctx.get(PNeedTransportRoutes.name),
            PNeedTransportSubway.name: self.ctx.get(PNeedTransportSubway.name),
            PMoscowOnly.name: self.ctx.get(PMoscowOnly.name),
            const.LABELS: labels
        }
        self.fill_subtask_parameters(subtask_parameters)
        logging.info('create subtask for convert graph for %s' % description)
        return self.create_subtask(task_type='SAAS_GARDEN_GRAPH_CONVERT',
                            description='Convert %s to %s, created by %s %s' % (description, res_type, type, self.id),
                            input_parameters=subtask_parameters,
                            execution_space=300000,
                            ram=81920
                            )

    def get_garden_export(self, region, environment=None):
        garden_environment = environment or self.ctx.get(PGardenEnvironment.name)
        attrs = {
            'garden_environment': garden_environment,
            'region': region
        }
        return apihelpers.get_last_resource_with_attrs(resource_type=resource_types.RTYSERVER_GARDEN_EXPORT, attrs=attrs, all_attrs=True)

    def import_from_maps_garden(self):
        tasks_count = 0
        sb_resource_rus = [
            self.get_garden_export('cis1'),
            self.get_garden_export('cis2'),
            self.get_garden_export('aao'),
            self.get_garden_export('eu1'),
        ]
        if sb_resource_rus is not None:
            task = self.build_convert_task(sb_resource_rus, resource_types.RTYSERVER_GRAPH_DIR_PRODUCTION_RUS)
            tasks_count += 1 if task else 0

        sb_resource_tur = [
            self.get_garden_export('tr'),
        ]
        if sb_resource_tur is not None:
            task = self.build_convert_task(sb_resource_tur, resource_types.RTYSERVER_GRAPH_DIR_PRODUCTION_TURKEY)
            tasks_count += 1 if task else 0

        sb_resource_AND = [
            self.get_garden_export('eu1'),
            self.get_garden_export('eu3'),
            self.get_garden_export('eu4'),
        ]
        if sb_resource_AND is not None:
            task = self.build_convert_task(sb_resource_AND, resource_types.RTYSERVER_GRAPH_DIR_PRODUCTION_AND)
            tasks_count += 1 if task else 0

        # sb_resource_europe = [
        #     self.get_garden_export('cis1'),
        #     self.get_garden_export('eu1'),
        #     self.get_garden_export('eu3'),
        #     self.get_garden_export('eu4'),
        # ]
        # if sb_resource_europe is not None:
        #   task = self.build_convert_task(sb_resource_europe, resources.RTYSERVER_GRAPH_DIR_PRODUCTION_EUROPE)
        #   tasks_count += 1 if task else 0

        sb_resource_eu1 = [
            self.get_garden_export('eu1'),
        ]
        if sb_resource_eu1 is not None:
            task = self.build_convert_task(sb_resource_eu1, resources.RTYSERVER_GRAPH_DIR_PRODUCTION_EU1)
            tasks_count += 1 if task else 0

        sb_resource_eu2 = [
            self.get_garden_export('eu2'),
        ]
        if sb_resource_eu2 is not None:
            task = self.build_convert_task(sb_resource_eu2, resources.RTYSERVER_GRAPH_DIR_PRODUCTION_EU2)
            tasks_count += 1 if task else 0

        return tasks_count

    def merge_graphs(self):
        for new_resource_type, parts_types in self.MERGE_RESOURCE_TYPE_BY_PARTS.items():
            logging.info('Merge %s to %s' % (parts_types, str(new_resource_type)))
            parts_ids = []
            parts_types_str = []
            ok = True
            for part_type in parts_types:
                part = apihelpers.get_last_resource(part_type, {'attribute_name': SaasGardenGraphConverter.AUTO_GENERATED_ATTR, 'attribute_value': 'yes'})
                if not part:
                    logging.error('there is no resource with type %s' % part_type)
                    ok = False
                    break
                parts_ids.append(part.id)
                part_name = '%s (%s)' % (get_graph_name(part.attributes), part.id)
                parts_types_str.append(part_name)
            if not ok:
                continue
            parts_ids = sorted(parts_ids)
            previes = apihelpers.get_resources_with_attribute(new_resource_type, SaasGardenGraphConverter.INPUT_GRAPHS_IDS_ATTR, str(parts_ids), status='READY')
            if previes:
                logging.info('merge for graphs %s already exists: %s' % (str(parts_ids), previes))
                continue
            subtask_parameters = {
                PSaasGraphs.name: parts_ids,
                PResultResourceType.name: str(new_resource_type),
                PMailForProblemReport.name: self.ctx.get(PMailForMergeProblemReport.name),
                PMode.name: 'merge'
            }
            self.fill_subtask_parameters(subtask_parameters)
            logging.info('create subtask for merge graphs from %s to %s' % (parts_ids, str(new_resource_type)))
            self.create_subtask(task_type='SAAS_GARDEN_GRAPH_CONVERT',
                                description='Merge %s to %s, created by %s %s' % (parts_types_str, str(new_resource_type), type, self.id),
                                input_parameters=subtask_parameters,
                                execution_space=300000,
                                ram=81920
                                )

    def traffic_history(self):
        for source_resource_type, target_resource_type in self.TRAFFIC_HISTORY_RESOURCE_TYPES.items():
            logging.info('building traffic history for %s' % source_resource_type)

            # Process only stable resources.
            graph_resource = apihelpers.get_last_resource_with_attribute(source_resource_type,
                                                                         'released_to_stable',
                                                                         'yes'
                                                                         )
            if graph_resource:
                self.build_traffic_history(graph_resource, target_resource_type)
            else:
                logging.info('stable {} does not exist'.format(source_resource_type))

        for region in ['aao-cis1-cis2-eu1']:
            logging.info('building traffic history for ABSTRACT {}', region)

            attrs = {
                'released_to_stable': 'yes',
                'region': region,
                'transport_routes': str(self.ctx.get(PNeedTransportRoutes.name)),
                'transport_stops': str(self.ctx.get(PNeedTransportStops.name)),
                'transport_subway': str(self.ctx.get(PNeedTransportSubway.name))
            }
            graph_resource = apihelpers.get_last_resource_with_attrs(
                resources.RTYSERVER_GRAPH_ABSTRACT,
                attrs=attrs,
                all_attrs=True
            )
            if graph_resource:
                self.build_traffic_history(graph_resource, resources.RTYSERVER_GRAPH_ABSTRACT)
            else:
                logging.info('stable ABSTRACT {} resourse does not exist'.format(region))

    def build_traffic_history(self, graph_resource, target_resource_type):
            # Don't process the same graph twice.
        graph_with_history_resource_id = graph_resource.attributes.get(
            SaasRoadsGraphTrafficHistoryBuilder.GRAPH_WITH_HISTORY_ID_ATTRIBUTE,
        )
        if graph_with_history_resource_id:
            logging.info('traffic history for %s has been already built in %s',
                         graph_resource.id,
                         graph_with_history_resource_id)
            return

        labels = {}
        labels['region'] = graph_resource.attributes.get('region')
        labels['garden_environment'] = graph_resource.attributes.get('garden_environment')

        subtask_parameters = {
            'kill_timeout': SaasRoadsGraphTrafficHistoryBuilder.TIMEOUT,
            PAutoRelease.name: self.ctx.get(PAutoRelease.name),
            PSaasGraphs.name: [graph_resource.id],
            PResultResourceType.name: str(target_resource_type),
            PMode.name: 'traffic_history',
            PNeedTransportStops.name: self.ctx.get(PNeedTransportStops.name),
            PNeedTransportRoutes.name: self.ctx.get(PNeedTransportRoutes.name),
            PNeedTransportSubway.name: self.ctx.get(PNeedTransportSubway.name),
            const.LABELS: labels
        }
        self.fill_subtask_parameters(subtask_parameters)
        logging.info('create subtask for traffic history for %s' % graph_resource.id)
        logging.info('subtask parameters: %s' % subtask_parameters)
        self.create_subtask(task_type='SAAS_GARDEN_GRAPH_CONVERT',
                            description='Traffic history for %s' % graph_resource.id,
                            input_parameters=subtask_parameters,
                            execution_space=300000,
                            ram=81920
                            )

    def on_execute(self):
        if self.ctx.get(PImportEnabled.name, True):
            imports_count = self.import_from_maps_garden()
            if imports_count > 0:
                self.wait_subtasks_with_children()
        if self.ctx.get(PMergeEnabled.name, True):
            self.merge_graphs()
            self.wait_subtasks_with_children()
        if self.ctx.get(PTrafficHistoryEnabled.name, True):
            self.traffic_history()


__Task__ = SaasGardenHelper
