# -*- coding: utf-8 -*-

from sandbox.sandboxsdk.task import SandboxTask

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import ResourceSelector, SandboxStringParameter
from sandbox.projects import resource_types
from sandbox.projects.common import apihelpers

import datetime
import logging
import time
import os


class ConverterResourceParameter(ResourceSelector):
    resource_type = resource_types.RTYSERVER_UTILS_ROADS_GRAPH
    required = False
    name, description = 'converter_resource_id', 'roads_graph binary'


class SaasServiceNameParameter(SandboxStringParameter):
    required = False
    name, description = 'saas_service_name', 'GeoSaaS service for matching'
    default_value = 'maps_graph_russia'


class YTServerParameter(SandboxStringParameter):
    required = False
    name, description = 'yt_server', 'YT server'
    default_value = 'hahn.yt.yandex.net'


class YTSourceTableParameter(SandboxStringParameter):
    required = True
    name, description = 'yt_src_table', 'YT source table'


class YTDestinationTableParameter(SandboxStringParameter):
    required = True
    name, description = 'yt_dst_table', 'YT destination table'


class SaasRoadsGraphMatcher(SandboxTask):
    type = 'SAAS_ROADS_GRAPH_MATCHER'

    input_parameters = [
        ConverterResourceParameter,
        SaasServiceNameParameter,
        YTServerParameter,
        YTSourceTableParameter,
        YTDestinationTableParameter,
    ]

    execution_space = 10000
    mails = ['svshevtsov']

    def get_parameter_value(self, resource):
        value = self.ctx[resource.name]
        if not value:
            if resource.required:
                raise errors.SandboxTaskFailureError("cannot get parameter value: %s" % resource.name)
            else:
                return resource.default_value
        return value

    def get_placeholders(self):
        now = datetime.datetime.now()
        utc_now = datetime.datetime.utcnow()
        yesterday = now - datetime.timedelta(days=1)
        two_days_ago = now - datetime.timedelta(days=2)
        week_ago = now - datetime.timedelta(days=7)
        return {
            '%YEAR%': now.year,
            '%MONTH%': '%02d' % now.month,
            '%DAY%': '%02d' % now.day,
            '%HOUR%': '%02d' % now.hour,
            '%MINUTE%': '%02d' % now.minute,
            '%SECOND%': '%02d' % now.second,
            '%UTC_YEAR%': utc_now.year,
            '%UTC_MONTH%': '%02d' % utc_now.month,
            '%UTC_DAY%': '%02d' % utc_now.day,
            '%UTC_HOUR%': '%02d' % utc_now.hour,
            '%UTC_MINUTE%': '%02d' % utc_now.minute,
            '%UTC_SECOND%': '%02d' % utc_now.second,
            '%YESTERDAY_YEAR%': yesterday.year,
            '%YESTERDAY_MONTH%': '%02d' % yesterday.month,
            '%YESTERDAY_DAY%': '%02d' % yesterday.day,
            '%TWO_DAYS_AGO_YEAR%': two_days_ago.year,
            '%TWO_DAYS_AGO_MONTH%': '%02d' % two_days_ago.month,
            '%TWO_DAYS_AGO_DAY%': '%02d' % two_days_ago.day,
            '%WEEK_AGO_YEAR%': week_ago.year,
            '%WEEK_AGO_MONTH%': '%02d' % week_ago.month,
            '%WEEK_AGO_DAY%': '%02d' % week_ago.day,
            '%TIMESTAMP%': int(time.mktime(now.timetuple())),
            '%USER%': self.owner,
            '%OWNER%': self.owner
        }

    def replace_placeholders(self, source, placeholders):
        result = source
        for key, value in placeholders.iteritems():
            result = result.replace(key, str(value))
        return result

    def get_converter(self):
        converter_resource_id = self.ctx[ConverterResourceParameter.name]
        if not converter_resource_id:
            converter = apihelpers.get_last_released_resource(resource_types.RTYSERVER_UTILS_ROADS_GRAPH)
            if not converter:
                raise errors.SandboxTaskFailureError("I don't know witch roads_graph to use")
            logging.info('converter selected: %s' % converter)
            converter_resource_id = converter.id

        logging.info('converter resource id %s' % converter_resource_id)
        return channel.task.sync_resource(converter_resource_id)

    def on_execute(self):
        placeholders = self.get_placeholders()

        saas_service = self.get_parameter_value(SaasServiceNameParameter)

        server = self.get_parameter_value(YTServerParameter)
        source = self.replace_placeholders(
            self.get_parameter_value(YTSourceTableParameter),
            placeholders
        )
        destination = self.replace_placeholders(
            self.get_parameter_value(YTDestinationTableParameter),
            placeholders
        )

        yt_token = self.get_vault_data(self.owner, 'YT_TOKEN')
        if yt_token:
            os.environ['YT_TOKEN'] = yt_token

        cmd = [
                self.get_converter(), 'features', 'prematch',
                '--saas-service', saas_service,
                '--yt-server', server,
                '--yt-src-table', source,
                '--yt-dst-table', destination,
        ]
        run_process(cmd, 'roads_graph')


__Task__ = SaasRoadsGraphMatcher
