# -*- coding: utf-8 -*-

import os
import logging

from sandbox import sdk2
from sandbox.common import errors

import sandbox.common.types.resource as ctr

from sandbox.projects.saas.common.resources import SAAS_UTIL_STANDALONE_INDEXER
from sandbox.projects.SaasStandaloneIndexerYt import SaasStandaloneIndexerYtConfigs


def dict_to_dsv(d):
    for k, v in d.iteritems():
        return '{}={}'.format(k, v)


class SaasStandaloneIndexerYtTask(sdk2.Task):
    """
    Запускает standalone_indexer для варки готовых шардов
    https://wiki.yandex-team.ru/jandekspoisk/saas/saaskv/welldoneshardsbyyt/
    """

    class Parameters(sdk2.Task.Parameters):
        yt_vault_token = sdk2.parameters.String(
            'Your yt token name in vault',
            default='yt-token',
            required=True
        )

        with sdk2.parameters.Group('Required parameters') as required_parameters:
            standalone_indexer = sdk2.parameters.Resource(
                'standalone_indexer',
                resource_type=SAAS_UTIL_STANDALONE_INDEXER,
                state=(ctr.State.READY),
                required=True
            )
            proxy = sdk2.parameters.String(
                'PROXY',
                description='YT cluster to run the indexer',
                required=True
            )
            src = sdk2.parameters.String(
                'SRC_TABLE',
                description='path to YT table with documents for indexing (schema depends on PROCESSOR)\n'
                            'can be supplited multiple times for more than one input table',
                required=True
            )
            dst_dir = sdk2.parameters.String(
                'DST_DIR',
                description='path to YT dir for indexing results',
                required=True
            )
            service = sdk2.parameters.String(
                'SERVICE',
                description='name of the SaaS service',
                required=True
            )
            configs = sdk2.parameters.Resource(
                'Configs',
                description='query-language rtyserver.conf-common searchmap.json',
                resource_type=SaasStandaloneIndexerYtConfigs,
                state=(ctr.State.READY),
                required=True
            )

        with sdk2.parameters.Group('Optional parameters') as optional_parameters:
            t = sdk2.parameters.Bool(
                '-t',
                description='test validate config file and exit'
            )
            V = sdk2.parameters.Dict(
                '-V',
                sdk2.parameters.String,
                description='set config Lua variable key to value',
            )
            P = sdk2.parameters.Dict(
                '-P',
                sdk2.parameters.String,
                description='patch config field key to value',
            )

            dst = sdk2.parameters.String(
                'DST_TABLE',
                description='path to YT table to store index (default: "DST_DIR/data/SERVICE/TIMESTAMP.blob")',
            )
            tmp_dir = sdk2.parameters.String(
                'TMP_DIR',
                description='path to YT dir for temp tables (default: "DST_DIR/tmp/SERVICE_TIMESTAMP/")',
            )
            publish = sdk2.parameters.Bool(
                '--publish',
                description='publish index in the snapshot manager (default: "false")'
            )

            publish_manager = sdk2.parameters.String(
                'PUB_MGR',
                description='snapshot manager type: yt, zookeeper (default: "yt")',
            )
            publish_server = sdk2.parameters.String(
                'PUB_SRV',
                description='snapshot server: YT proxy or ZK hosts (default: "PROXY" when PUB_MGR="yt")',
            )
            publish_path = sdk2.parameters.String(
                'PUB_PATH',
                description='ypath or znode to publish results (default: "DST_DIR/testing/SERVICE" when PUB_MGR="yt")',
            )

            timestamp = sdk2.parameters.Integer(
                'TIMESTAMP',
                description='modification timestamp for documents(default: now)',
                default=None,
            )
            processor = sdk2.parameters.String(
                'PROCESSOR',
                description='processor to convert rows of SRC_TABLE into SaaS TActions (default: "personal")',
            )
            segment_size = sdk2.parameters.Integer(
                'SEGMENT',
                description='max documents per segment (default: CONFIG.MergerConfig.MaxDocumentsToMerge)',
                default=None,
            )

            keep_temps = sdk2.parameters.Bool(
                '--keep-temps',
                description='don\'t remove TMP_DIR on successful completion (default: "false")',
            )
            force = sdk2.parameters.Bool(
                '--force',
                description='overwrite DST_TABLE if it already exists (default: "false")',
            )
            resume = sdk2.parameters.Bool(
                '--resume',
                description='pick up intermediate results from TMP_DIR, use with --timestamp option (default: "false")',
            )

            verbose = sdk2.parameters.Bool(
                '--verbose',
                description='print progress to stderr (default: "false")',
            )
            dry_run = sdk2.parameters.Bool(
                '--dry-run',
                description='only validate YT specs, don\'t run any operations (default: "false")',
            )

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        assert self.Parameters.service == self.Parameters.configs.service_name

        configs = sdk2.ResourceData(self.Parameters.configs)
        standalone_indexer = sdk2.ResourceData(self.Parameters.standalone_indexer)

        cmd = [
            str(standalone_indexer.path), 'yt',
            os.path.join(str(configs.path), 'rtyserver.conf-common'),
            '--proxy', self.Parameters.proxy,
            '--src', self.Parameters.src,
            '--dst-dir', self.Parameters.dst_dir,
            '--service', self.Parameters.service,
            '--searchmap', os.path.join(str(configs.path), 'searchmap.json'),
        ]

        if self.Parameters.t:
            cmd.append('-t')

        if self.Parameters.V:
            for value in dict_to_dsv(self.Parameters.V):
                cmd += ['-V', value]

        if self.Parameters.P:
            for value in dict_to_dsv(self.Parameters.P):
                cmd += ['-P', value]

        if self.Parameters.dst:
            cmd += ['--dst', self.Parameters.dst]

        if self.Parameters.tmp_dir:
            cmd += ['--tmp-dir', self.Parameters.tmp_dir]

        if self.Parameters.publish:
            cmd.append('--publish')

        if self.Parameters.publish_manager:
            cmd += ['--publish-manager', self.Parameters.publish_manager]

        if self.Parameters.publish_server:
            cmd += ['--publish-server', self.Parameters.publish_server]

        if self.Parameters.publish_path:
            cmd += ['--publish-path', self.Parameters.publish_path]

        if self.Parameters.timestamp:
            cmd += ['--timestamp', str(self.Parameters.timestamp)]

        if self.Parameters.processor:
            cmd += ['--processor', self.Parameters.processor]

        if self.Parameters.segment_size:
            cmd += ['--segment-size', str(self.Parameters.segment_size)]

        if self.Parameters.keep_temps:
            cmd.append('--keep-temps')

        if self.Parameters.force:
            cmd.append('--force')

        if self.Parameters.resume:
            cmd.append('--resume')

        if self.Parameters.verbose:
            cmd.append('--verbose')

        if self.Parameters.dry_run:
            cmd.append('--dry-run')

        logging.info(' '.join(cmd))

        with sdk2.helpers.ProcessLog(self, logger='run standalone_indexer') as pl:
            env = os.environ.copy()
            env['YT_TOKEN'] = sdk2.Vault.data(self.owner, self.Parameters.yt_vault_token)
            ret = sdk2.helpers.subprocess.Popen(cmd, stdout=pl.stdout, stderr=pl.stderr, env=env).wait()
            if ret:
                raise errors.TaskError('run is failed')
