# -*- coding: utf-8 -*-

import os

from sandbox.projects import resource_types
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import make_folder


ROOT = 'arcadia:/arc/trunk/arcadia/samogon/test'


def full(path):
    return os.path.join(ROOT, path)


ANOTHER_SERVANT_DIR = full('services/another')
BOMBER_SERVANT_DIR = full('services/bomber')
IMPORT_SERVANT_DIR = full('services/import_srv')
SIMPLE_SERVANT_DIR = full('services/simple')
RANDSENSORS_SERVANT_DIR = full('services/randsensors')
TOKEN = full('token.oauth')
SYSDEPS_SERVANT_DIR = full('services/sysdeps')
GENERAL_DIR = full('general')
PLUGIN_DIR = full('plugin')


def new_name(file_name, new_dir):
    return os.path.join(new_dir, os.path.basename(file_name))


class SamogonTestPackage(SandboxTask):
    '''
        create Samogon test package.
    '''

    type = 'SAMOGON_TEST_PACKAGE'

    def create_archive(self, source, dest):
        workdir = os.path.dirname(source)
        run_process(
            'tar -czf %s %s' % (dest, os.path.basename(source)),
            shell=True,
            log_prefix='tar',
            work_dir=workdir
        )

    def on_execute(self):
        tmp_dir = make_folder(os.path.join(self.abs_path(), 'temp'))
        new_dir = make_folder(os.path.join(self.abs_path(), 'samogon-test-package'))
        dest_dir = os.path.join(new_dir, 'crossplatform')
        make_folder(dest_dir)

        another_files = [ANOTHER_SERVANT_DIR, GENERAL_DIR]
        another_dir = make_folder(os.path.join(tmp_dir, 'another'))
        for file in another_files:
            Arcadia.export(file, new_name(file, another_dir))
        self.create_archive(another_dir, os.path.join(dest_dir, 'another.tgz'))

        bomber_files = [BOMBER_SERVANT_DIR, GENERAL_DIR]
        bomber_dir = make_folder(os.path.join(tmp_dir, 'bomber'))
        for file in bomber_files:
            Arcadia.export(file, new_name(file, bomber_dir))
        self.create_archive(bomber_dir, os.path.join(dest_dir, 'bomber.tgz'))

        import_srv_files = [IMPORT_SERVANT_DIR, GENERAL_DIR]
        import_srv_dir = make_folder(os.path.join(tmp_dir, 'import_srv'))
        for file in import_srv_files:
            Arcadia.export(file, new_name(file, import_srv_dir))
        self.create_archive(import_srv_dir, os.path.join(dest_dir, 'import_srv.tgz'))

        simple_files = [SIMPLE_SERVANT_DIR, GENERAL_DIR]
        simple_dir = make_folder(os.path.join(tmp_dir, 'simple'))
        for file in simple_files:
            Arcadia.export(file, new_name(file, simple_dir))
        self.create_archive(simple_dir, os.path.join(dest_dir, 'simple.tgz'))

        sysdeps_files = [SYSDEPS_SERVANT_DIR, GENERAL_DIR]
        sysdeps_dir = make_folder(os.path.join(tmp_dir, 'sysdeps'))
        for file in sysdeps_files:
            Arcadia.export(file, new_name(file, sysdeps_dir))
        self.create_archive(sysdeps_dir, os.path.join(dest_dir, 'sysdeps.tgz'))

        randsens_files = [RANDSENSORS_SERVANT_DIR, GENERAL_DIR]
        randsens_dir = make_folder(os.path.join(tmp_dir, 'randsensors'))
        for file in randsens_files:
            Arcadia.export(file, new_name(file, randsens_dir))
        self.create_archive(randsens_dir, os.path.join(dest_dir, 'randsensors.tgz'))

        plugin_dir = os.path.join(new_dir, 'plugin')
        Arcadia.export(PLUGIN_DIR, plugin_dir)

        archive = os.path.join(tmp_dir, 'samogon_test.tgz')
        self.create_archive(new_dir, archive)

        self.create_resource(
            description="Samogon test package",
            resource_path=archive,
            resource_type=resource_types.SAMOGON_TEST_PACKAGE,
            attributes={
                "ttl": 180,
            }
        )


__Task__ = SamogonTestPackage
