# -*- coding: utf-8 -*-

import json

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import SvnEnvironment

import sandbox.sandboxsdk.parameters as sdk_parameters

from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import process

from sandbox.projects.PSUtil import calculate_diff, create_file_with_content


class SanitizerCompareFuncTestResults(SandboxTask):
    class OldAnsResourceId(sdk_parameters.ResourceSelector):
        name = 'old_ans_resource_id'
        description = 'Old answers'
        resource_type = resource_types.SANITIZER_FUNC_TEST_RESULT
        required = True

    class NewAnsResourceId(sdk_parameters.ResourceSelector):
        name = 'new_ans_resource_id'
        description = 'New answers'
        resource_type = resource_types.SANITIZER_FUNC_TEST_RESULT
        required = True

    class TestsResourceId(sdk_parameters.ResourceSelector):
        name = 'tests_resource_id'
        description = 'Sanitizer tests'
        resource_type = resource_types.IEX_TESTS
        required = True

    type = 'SANITIZER_COMPARE_FUNC_TEST_RESULTS'
    input_parameters = [
        OldAnsResourceId,
        NewAnsResourceId,
        TestsResourceId
    ]

    client_tags = ctc.Tag.LINUX_PRECISE
    execution_space = 1000
    environment = (SvnEnvironment(), )

    def __init__(self, task_id=0):
        SandboxTask.__init__(self, task_id)

    def on_execute(self):
        diff_dir = self.abs_path('diff')
        make_folder(diff_dir)
        self.create_resource(
            description=self.descr,
            resource_type=resource_types.SANITIZER_DIFF_FUNC_RESULTS,
            resource_path=diff_dir,
            attributes={'ttl': 100, 'backup_task': True}
        )

        tests_url = (channel.sandbox.get_resource(self.ctx[self.TestsResourceId.name])).url
        diff_path = diff_dir + '/sanitizer.diffs.html'
        create_file_with_content(diff_path, '<head><meta charset="UTF-8"></head>\n')
        fails_diff_path = diff_dir + '/sanitizer.only_diffs.html'
        create_file_with_content(fails_diff_path, '<head><meta charset="UTF-8"></head>\n')

        tests_dir = self.sync_resource(self.ctx['tests_resource_id'])
        tests_info = json.loads(open(tests_dir + '/tests_data.json').read())
        has_diffs = False
        for test_info in tests_info:
            new_answer_path = '{}/{}.txt'.format(self.sync_resource(self.ctx[self.NewAnsResourceId.name]), test_info['id'])
            old_answer_path = '{}/{}.txt'.format(self.sync_resource(self.ctx[self.OldAnsResourceId.name]), test_info['id'])
            log = '<b>{id}::<a target="_blank" href="{url}&relpath=/doc/{id}.html">Doc</a>::{query}</b>'.format(
                id=test_info['id'],
                url=tests_url,
                query=test_info['query']
            )
            c = calculate_diff(log, 'sanitizer', old_answer_path, new_answer_path, diff_path, fails_diff_path)
            has_diffs = has_diffs or c

        if has_diffs:
            process.run_process(['mv', diff_path, diff_dir + '/!sanitizer.diffs.html'])
            process.run_process(['mv', fails_diff_path, diff_dir + '/!sanitizer.only_diffs.html'])

            self.ctx['has_diff'] = json.dumps(['sanitizer'])
            self.set_info('Дифф есть')
        else:
            self.ctx['has_diff'] = json.dumps([])
            self.set_info('Диффа нет')


__Task__ = SanitizerCompareFuncTestResults
