# -*- coding: utf-8 -*-

import logging
import json
import os
import time
from os import path, system

import sandbox.common.types.client as ctc

from sandbox.projects.resource_types import SANITIZER, IEX_TESTS
from sandbox.projects.common.FuncTestSuiteBase import FuncTestSuiteBase
from sandbox.sandboxsdk.environments import SvnEnvironment
from sandbox.sandboxsdk.paths import make_folder

import sandbox.sandboxsdk.parameters as sdk_parameters

from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.PSUtil import append_text, read_text, replace, comment_section_in_arcadia_conf, get_port


class SanitizerTestFunc(FuncTestSuiteBase):
    class SanitizerResourceId(sdk_parameters.ResourceSelector):
        name = 'sanitizer_resource_id'
        description = 'sanitizer binary'
        resource_type = SANITIZER
        required = True

    class TestsResourceId(sdk_parameters.ResourceSelector):
        name = 'tests_resource_id'
        description = 'Func tests'
        resource_type = IEX_TESTS
        required = True

    class ConfigDirSvn(sdk_parameters.SandboxStringParameter):
        name = 'config_dir_svn'
        description = 'Path to svn with daemon config'
        default_value = 'arcadia:/arc/trunk/arcadia/yweb/mail/wmiarc/pack'

    class DiffScriptSvn(sdk_parameters.SandboxStringParameter):
        name = 'diff_script_svn'
        description = 'Path to svn with script diff.sh'
        default_value = 'arcadia:/arc/trunk/arcadia/sandbox/projects/SanitizerTestFunc/diff.sh'

    type = 'SANITIZER_TEST_FUNC'
    input_parameters = [
        SanitizerResourceId,
        TestsResourceId,
        ConfigDirSvn,
        DiffScriptSvn
    ]
    environment = (SvnEnvironment(), )

    execution_space = 3000
    client_tags = ctc.Tag.LINUX_PRECISE

    def __init__(self, task_id=0):
        FuncTestSuiteBase.__init__(self, task_id)

    def download_data(self):
        configs_dir = os.path.join(self.abs_path(), 'configs')
        make_folder(configs_dir)
        Arcadia.export(self.ctx['config_dir_svn'], configs_dir)
        config_path = os.path.join(configs_dir, 'config')
        log_path = str(os.path.join(self.log_path(), 'sanitizer.log'))
        Arcadia.export(self.ctx['diff_script_svn'], self.abs_path())
        diff_script_path = os.path.join(self.abs_path(), self.ctx['diff_script_svn'].split('/')[-1])
        binary_res_path = self.sync_resource(self.ctx['sanitizer_resource_id'])
        SanitizerTestFunc.edit_config(config_path, configs_dir, log_path)
        return binary_res_path, config_path, diff_script_path, log_path

    @staticmethod
    def edit_config(config_path, configs_dir, log_path):
        replace(config_path, '/var/log/sanitizer/sanitizer.log', log_path)
        replace(config_path, '/var/run', configs_dir)
        replace(config_path, '/etc/sanitizer', configs_dir)
        replace(config_path, 'ReqResLog : off', 'ReqResLog : on')
        replace(config_path, 'Pid :', '#Pid :')
        replace(config_path, get_port(config_path), '10110')
        comment_section_in_arcadia_conf(config_path, 'Rca')
        logging.info('Config:\n' + read_text(config_path))

    def test(self, port, core_path):
        tests_dir = self.sync_resource(self.ctx['tests_resource_id'])
        tests_info = json.loads(open(tests_dir + '/tests_data.json').read())

        # send_requests
        answs_dir = self.abs_path('ans')
        make_folder(answs_dir)
        for test_info in tests_info:
            doc_path = tests_dir + '/doc/' + str(test_info['id']) + '.html'
            ans_path = answs_dir + '/' + str(test_info['id']) + '.txt'
            if path.exists(doc_path):
                wget_arg = test_info['wget_arg'] if ('wget_arg' in test_info) and (test_info['wget_arg']) else ''
                system("wget --max-redirect 0 -t 1 -T 5 -O '" + ans_path + "' " + wget_arg + " 'localhost:" + str(port)
                       + test_info['query'] + "' --post-file='" + doc_path + "'")
            else:
                append_text(core_path, 'ERROR: Document ' + doc_path + ' has no exists\n')
        r = self.create_resource(description=self.descr, resource_path=answs_dir,
                                 resource_type='SANITIZER_FUNC_TEST_RESULT')
        self.mark_resource_ready(r)

    def on_execute(self):
        binary_res_path, config_path, diff_script, log_path = self.download_data()

        # run sanitizer
        run_process([binary_res_path + '/wmiarc', config_path], log_prefix='run_sanitizer', wait=False)
        time.sleep(10)  # sec

        # test Sanitizer
        core_path = self.log_path('ERRORS')
        self.test(get_port(config_path), core_path)


__Task__ = SanitizerTestFunc
