# -*- coding: utf-8 -*-

import datetime
import inspect
import time

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxIntegerParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.sandboxapi import ARCH_ANY
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects.common.nanny import nanny

import sandbox.projects.resource_types


class ShardTemplate(SandboxStringParameter):
    name = 'shard_template'
    description = 'Shard template (i.e. imgfth-000-00000000-000000)'
    required = True


class ShardState(SandboxStringParameter):
    name = 'shard_state'
    description = 'Shard state'
    required = True


class ShardCount(SandboxIntegerParameter):
    name = 'shard_count'
    description = 'Shard count'
    required = True


class ShardmapType(SandboxStringParameter):
    def create_choises():
        result = []
        for name, resource in inspect.getmembers(sandbox.projects.resource_types):
            if inspect.isclass(resource):
                if hasattr(resource, 'shardmap') and resource.shardmap:
                    result.append((name, name))
        return result

    name = 'shardmap_type'
    description = 'Shardmap type'
    choices = create_choises()
    required = True


class ShardTier(SandboxStringParameter):
    name = 'shard_tier'
    description = 'Shard tier'
    required = True


class SearchCreateShardMap(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Создание поискового шардмапа
    """

    type = 'SEARCH_CREATE_SHARDMAP'

    cores = 1

    input_parameters = [ShardTemplate, ShardState, ShardCount, ShardmapType, ShardTier]

    _RESOURCE_ID = 'resource_id'

    def write_shardmap(self, shardmap_file_path):
        with open(shardmap_file_path, 'w') as shardmap_file:
            template = self.ctx.get(ShardTemplate.name)
            shard_count = self.ctx.get(ShardCount.name)
            shard_state = self.ctx.get(ShardState.name)
            shard_tier = self.ctx.get(ShardTier.name)

            parts = template.split('-')
            shard_name = parts[0]
            shard_num_template = parts[1]
            shard_ts_template = '-'.join(parts[2:])

            for shard in range(0, int(shard_count)):
                shard_str = str(shard).zfill(len(shard_num_template))
                shardmap_file.write('{0}-{1}-{2} {0}-{1}-{3} {4}\n'.format(shard_name, shard_str, shard_ts_template,
                                                                           shard_state, shard_tier))

    def on_enqueue(self):
        shardmap_file_name = 'search_shardmap'
        shard_state = self.ctx.get(ShardState.name)
        shardmap_timestamp = str(int(time.mktime(datetime.datetime.strptime(shard_state, '%Y%m%d-%H%M%S').timetuple())))
        resource = self.create_resource(
            self.descr,
            shardmap_file_name,
            self.ctx.get(ShardmapType.name),
            arch=ARCH_ANY,
            attributes={
                'shard_state': shard_state,
                'shardmap_timestamp': shardmap_timestamp
            }
        )

        self.ctx[self._RESOURCE_ID] = resource.id

    def on_execute(self):
        resource = channel.sandbox.get_resource(self.ctx[self._RESOURCE_ID])

        self.write_shardmap(resource.path)

        channel.task.mark_resource_ready(resource)


__Task__ = SearchCreateShardMap
