# -*- coding: utf-8 -*-

import logging
import re
from requests import get

from sandbox.projects.SecDis.data_types import Item, ItemType
from sandbox.projects.SecDis.Collectors import BaseCollector


class Github(object):
    base_url = 'https://api.github.yandex-team.ru'

    def __init__(self, owner, repo, client_id, client_secret):
        self.owner = owner
        self.repo = repo
        self.client_id = client_id
        self.client_secret = client_secret

    def get_tags_list(self, prev_versions=set()):
        url = '{}/repos/{}/{}/tags'.format(self.base_url, self.owner, self.repo)
        pat = re.compile(r'<([^<>]*?)>; rel=\"next\"')

        payload = {
            'client_id': self.client_id,
            'client_secret': self.client_secret,
            'per_page': 100,
        }

        st_pat = re.compile(ur'^\w+\-\d+$')

        result = dict()
        running = True
        while running:
            r = get(url, params=payload)
            if r.status_code != 200:
                logging.info('[get_tags_list] Github {}/{} status_code={}'.format(self.owner, self.repo, r.status_code))
                break
            json_data = r.json()
            logging.info('[get_tags_list] Github {}/{} len={}'.format(self.owner, self.repo, len(json_data)))
            for tag_info in json_data:
                name = tag_info['name']
                if name in prev_versions:
                    running = False
                    break
                result[name] = list()
                commit_url = tag_info['commit']['url']
                commit_message = self.get_commit_info(commit_url)
                for line in commit_message.split('\n'):
                    line = line.strip()
                    if not line.startswith('*'):
                        continue
                    line = line[2:].strip()
                    key = line.split(' ', 1)[0]
                    if key.endswith(':'):
                        key = key[:-1]
                    if st_pat.match(key) is None:
                        continue
                    summary = line.split(' ', 1)[-1]
                    result[name].append(dict(key=key, summary=summary))

            m = pat.search(r.headers.get('Link', ''))
            if not m:
                running = False
                break
            url = m.group(1)

        return result

    def get_commit_info(self, url):
        payload = {
            'client_id': self.client_id,
            'client_secret': self.client_secret,
        }
        r = get(url, payload)
        if r.status_code != 200:
            logging.info('[get_commit_info] Github {}/{} status_code={}'.format(self.owner, self.repo, r.status_code))
            return None
        json_data = r.json()
        message = json_data['commit']['message']

        return message


class GeoRelCollector(BaseCollector):
    """ Service Security Discovery: Geo Releases Monitoring """
    collector_name = 'georel'
    input_types = (ItemType.REPO,)
    output_types = (ItemType.RELEASE,)

    def on_execute(self):
        project_id = self.Parameters.project_id
        auxiliary = self.load_auxiliary()
        logging.info("auxiliary %s" % auxiliary)
        if auxiliary is None:
            auxiliary = set()
        else:
            auxiliary = set(auxiliary)

        item_list = [Item.unserialize(item) for item in self.Parameters.item_list if item['type'] == ItemType.REPO]
        repo_url = item_list[0].get_value()

        github_client_id = self.get_vault('GithubClientIdSecDis')
        github_client_secret = self.get_vault('GithubClientSecretSecDis')
        m = re.search(r'https:\/\/github\.yandex-team\.ru\/([\w\-]+)\/([\w\-]+)(?:.*)', repo_url)
        owner = m.group(1)
        repo = m.group(2)
        r = Github(owner, repo, github_client_id, github_client_secret)

        tags_info = r.get_tags_list(auxiliary)

        for version, info in tags_info.items():
            new_item = Item(ItemType.RELEASE, str(project_id), version, list(), sub_issues=info)
            self.add_result(new_item, list())

        auxiliary.update(set(tags_info.keys()))
        self.save_auxiliary(list(auxiliary))
        self.save_result()
