import logging

from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.ssh import Key
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk import svn

from sandbox.projects.SecDis.data_types import Item, ItemType
from sandbox.projects.SecDis.Collectors import BaseCollector

from sandbox.projects.SecDis.Collectors.HandleCollector.nodejs import parse_nodejs
from sandbox.projects.SecDis.Collectors.HandleCollector.spring import parse_spring
from sandbox.projects.SecDis.Collectors.HandleCollector.fastcgi_daemon_conf import parse_fastcgi_daemon_conf
from sandbox.projects.SecDis.Collectors.HandleCollector.python_aiohttp import parse_python_aiohttp
from sandbox.projects.SecDis.Collectors.HandleCollector.python_django import parse_python_django
from sandbox.projects.SecDis.Collectors.HandleCollector.python_twisted_methods_dict import parse_python_twisted_methods_dict
from sandbox.projects.SecDis.Collectors.HandleCollector.php_symfony_route_annotation import parse_php_symfony_route_annotation
from sandbox.projects.SecDis.Collectors.HandleCollector.market_resolver import parse_node_resolver

lang2parser = {
    'nodejs': parse_nodejs,
    'spring': parse_spring,
    'fastcgi_daemon_conf': parse_fastcgi_daemon_conf,
    'python_aiohttp': parse_python_aiohttp,
    'python_django': parse_python_django,
    'python_twisted_methods_dict': parse_python_twisted_methods_dict,
    'php_symfony_route_annotation': parse_php_symfony_route_annotation,
    'market_resolver': parse_node_resolver
}


class HandleCollector(BaseCollector):
    """ Service Security Discovery: search for project handles in source code """
    collector_name = 'handle'
    input_types = (ItemType.REPO,)
    output_types = (ItemType.HANDLE,)

    def _git_clone(self, repo, branch=None):
        repo = repo.split('/', 3)[-1]
        folder = str(self.path('work'))
        if branch is None:
            branch_arg = ''
        else:
            branch_arg = '-b {branch}'.format(branch=branch)
        git_cmd = 'ssh-keyscan -H github.yandex-team.ru >> ~/.ssh/known_hosts && ' \
                  'git clone --depth 1 git@github.yandex-team.ru:{repo}.git ' \
                  '{branch_arg} {folder}'.format(repo=repo, branch_arg=branch_arg, folder=folder)

        try:
            with Key(self, self.owner, 'STOP_LEAK_SSH_KEY'), ProcessLog(self, logger=logging.getLogger('git')) as pl:
                sp.check_call(git_cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout)
            return folder
        except sp.CalledProcessError as e:
            logging.error('Run git clone error: %s', e)
            return None

    def _bb_clone(self, repo, branch=None):
        repo = repo.split('/', 3)[-1]
        folder = str(self.path('work'))
        if branch is None:
            branch_arg = ''
        else:
            branch_arg = '-b {branch}'.format(branch=branch)
        git_cmd = 'git clone --depth 1 https://{username}:{token}@bb.yandex-team.ru/scm/{repo}.git ' \
                    ' {branch_arg} {folder} '.format(username=self.get_vault('SECDIS_BB_USERNAME'), \
                    token=self.get_vault('SECDIS_BB_TOKEN'), repo=repo, branch_arg=branch_arg, folder=folder)
        with ProcessLog(self, logger=logging.getLogger('git')) as pl:
            sp.check_call(git_cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout)
        return folder

    def _arcadia_clone(self, repo):
        folder = str(self.path('work'))
        checkout_arcadia_from_url = 'arcadia:/arc/trunk/arcadia'
        revision = svn.Arcadia.parse_url(checkout_arcadia_from_url).revision
        svn.Arcadia.checkout(repo, folder, revision=revision)
        return folder

    def clone(self, item):
        repo = item.get_value()
        branch = item.fields.get('branch')

        if repo.startswith('https://github.yandex-team.ru/'):
            folder = self._git_clone(repo, branch)
        elif repo.startswith('https://bb.yandex-team.ru/'):
            folder = self._bb_clone(repo, branch)
        elif repo.startswith('https://a.yandex-team.ru/'):
            folder = self._arcadia_clone(repo)
        else:
            folder = None

        return folder

    def parse(self, item, folder):
        ancestor_id = item.get_id()
        project_id = self.Parameters.project_id
        lang = item.fields.get('lang')
        new_items = []

        if lang is None:
            for key, parser in lang2parser.items():
                new_items = parser(folder, project_id, ancestor_id)
                if new_items:
                    break
        else:
            parser = lang2parser.get(lang)
            if parser is None:
                logging.error('Language %s is not supported', lang)
                return None
            new_items = parser(folder, project_id, ancestor_id)

        return new_items

    def on_execute(self):
        item_list = [Item.unserialize(item) for item in self.Parameters.item_list]

        for item in item_list:
            folder = self.clone(item)
            if folder is None:
                logging.error('Error while cloning repo %s', item.get_value())
                continue

            new_items = self.parse(item, folder)
            if new_items is None:
                continue

            ancestor_id = item.get_id()
            for new_item in new_items:
                new_item.fields['repo'] = item.get_value()
                new_item.fields['branch'] = item.fields.get('branch')
                new_item.fields['handle'] = new_item.get_value()
                new_item.value = str(new_item.fields['repo']) + " : " + \
                                str(new_item.fields['branch']) + " : " + new_item.value
                self.add_result(new_item, ancestor_id)

        self.save_result()
