# -*- coding: utf-8 -*-

import logging
import datetime
import re

from sandbox.projects.SecDis.data_types import Item, ItemType
from sandbox.projects.SecDis.Collectors import BaseCollector

from sandbox.sandboxsdk.environments import PipEnvironment


class MailRelCollector(BaseCollector):
    """ Service Security Discovery: Mail Releases Startrack Queue Monitoring """
    collector_name = 'mailrel'
    input_types = (ItemType.QUEUE,)
    output_types = (ItemType.RELEASE,)

    class Requirements(BaseCollector.Requirements):
        environments = [PipEnvironment('startrek_client')]

    def get_releases_by_queue(self, queue_name, created_from):
        from startrek_client import Startrek

        access_token = self.get_vault('OAuthSecDis')
        client = Startrek(useragent="Mail Release Collector", base_url='https://st-api.yandex-team.ru',
                          token=access_token)
        issues = client.issues.find(
            filter={'queue': queue_name, 'updated': {'from': created_from}},
            order=['-created']
        )
        releases = []
        version_regex = re.compile(ur'\|\|\s+\*\*Версия\*\*\s+\|\s+\(\(https:\/\/st\.yandex-team\.ru\/\S+ ([\d\.]+)\)\)')
        commits_regex = re.compile(ur'\|\|\s+\*\*Изменения\*\*\s+\|\s+<{([\s\S]+?)}>')
        for index, issue in enumerate(issues):
            if index > 300:
                break
            comments = issue.comments.get_all()
            for comment in comments:
                author = comment.createdBy.login
                text = comment.text
                if author == 'robot-aqua-testpers'\
                        and u'Автоматический отчет о сборке пакета' in text \
                        and u'!!(green)SUCCESS!!' in text:
                    m = version_regex.search(text)
                    if m:
                        version = m.group(1)
                    else:
                        continue
                    commits = list()
                    m = commits_regex.search(text)
                    if m:
                        for line in m.group(1).split('\n'):
                            if line.startswith('(('):
                                commit = line.split(' ', 1)[0][2:]
                                description = line.split(' ', 1)[1][:-2]
                                commits.append(dict(commit=commit, summary=description))
                                logging.info("{} {} {}".format(issue.key, commit, description))
                    if version and commits:
                        data = {
                            'summary': version,
                            'key': issue.key,
                            'commits': commits,
                            'release_created_at': comment.createdAt,
                            'release_updated_at': comment.updatedAt
                        }
                        releases.append(data)

        logging.info("releases {}".format(releases))
        return releases


    def on_execute(self):
        project_id = self.Parameters.project_id
        auxiliary = self.load_auxiliary()
        logging.info("auxiliary %s" % auxiliary)
        if auxiliary is None:
            auxiliary = dict()
        default_start_datetime = datetime.datetime.now() - datetime.timedelta(days=30)
        default_start_date = '%d-%d-%d' % (default_start_datetime.year,
                                           default_start_datetime.month, default_start_datetime.day)
        start_date = auxiliary.get('start_date', default_start_date)

        queue_list = [Item.unserialize(item) for item in self.Parameters.item_list if item['type'] == ItemType.QUEUE]
        releases = list()
        for item in queue_list:
            queue = item.get_value()
            releases += self.get_releases_by_queue(queue, start_date)

        for release in releases:
            new_item = Item(ItemType.RELEASE, str(project_id),
                        release['summary'], list(),
                        created_at=release['release_created_at'], updated_at=release['release_updated_at'],
                        **release)
            self.add_result(new_item, list())

        datetime_now = datetime.datetime.now()
        current_start_date = '%d-%d-%d' % (datetime_now.year, datetime_now.month, datetime_now.day)
        self.save_auxiliary({'start_date': current_start_date})

        self.save_result()
