import logging
import datetime

from sandbox.projects.SecDis.data_types import Item, ItemType
from sandbox.projects.SecDis.Collectors import BaseCollector
from sandbox.sdk2.parameters import JSON
from sandbox.sandboxsdk.environments import PipEnvironment


class StTicketCollector(BaseCollector):
    """ Service Security Discovery: Monitoring Tickets from Startrack Queue """
    collector_name = 'st_ticket'
    input_types = (ItemType.QUEUE)
    output_types = (ItemType.ST_TICKET)

    class Requirements(BaseCollector.Requirements):
        environments = [PipEnvironment('startrek_client')]

    class Parameters(BaseCollector.Parameters):
        st_filter_options = JSON("filter_options", default={'resolution':'notEmpty()'}, required=False)

    def get_issues_by_queue(self, queue_name, updated_from, rel_tikets_limit=300, only_with_code=True, filter_options={}):
        from startrek_client import Startrek
        from startrek_client.exceptions import Forbidden

        access_token = self.get_vault('OAuthSecDis')
        client = Startrek(useragent="AppSec Discovery Tickets Collector", base_url='https://st-api.yandex-team.ru',
                          token=access_token)
        issues_filter = {'queue': queue_name, 'updated': {'from': updated_from}}
        for (filter_key, filter_value) in filter_options.items():
            issues_filter[filter_key] = filter_value
        issues = client.issues.find(
            filter=issues_filter,
            order=['-created']
        )
        result_issues = []
        for index, issue in enumerate(issues):
            if index >= rel_tikets_limit:
                break
            commits = []
            for remotelink in issue['remotelinks']:
                if remotelink.object.application.type == 'com.github':
                    # Pull-requests for github
                    (github_type, github_link) = remotelink.object.key.split(':')
                    if github_type == 'pull':
                        github_link_parts = github_link.split('/')
                        github_link_parts.insert(-1, 'pull')
                        if remotelink.object.application.id == 'ru.yandex.github.enterprise':
                            github_link_parts.insert(0, 'https://github.yandex-team.ru')
                        elif remotelink.object.application.id == 'com.github':
                            github_link_parts.insert(0, 'https://github.com')
                        github_link_parts.append('files')
                        github_link = '/'.join(github_link_parts)
                    commits.append(github_link)
                elif remotelink.object.application.type == 'ru.yandex.stash':
                    # Pull-requests for bb.yandex-team.ru
                    print(remotelink.object.key)
                    (bb_project, bb_repo, pr_num) = remotelink.object.key.split('/')
                    commits.append('/'.join(
                        ['https://bb.yandex-team.ru', 'projects', bb_project,
                        'repos', bb_repo, 'pull-requests', pr_num, 'diff']))
            if only_with_code and not(len(commits)):
                continue
            result_issues.append({
                'key': issue['key'],
                'summary': issue['summary'],
                'issue_created_at': issue['createdAt'],
                'issue_updated_at': issue['updatedAt'],
                'commits': commits
            })
        return result_issues

    def on_execute(self):
        project_id = self.Parameters.project_id
        auxiliary = self.load_auxiliary()
        logging.info("auxiliary %s" % auxiliary)
        if auxiliary is None:
            auxiliary = dict()
        default_start_datetime = datetime.datetime.now() - datetime.timedelta(days=21)
        default_start_date = '%d-%d-%d' % (default_start_datetime.year,
                                           default_start_datetime.month, default_start_datetime.day)
        start_date = auxiliary.get('start_date', default_start_date)

        queue_list = [Item.unserialize(item) for item in self.Parameters.item_list]

        for queue in queue_list:
            with_code = queue.fields.get('only_with_code', True)
            tickets = self.get_issues_by_queue(queue.get_value(), start_date, only_with_code=with_code,\
                filter_options=self.Parameters.st_filter_options)
            for st_ticket in tickets:
                new_st_ticket_item = Item(ItemType.ST_TICKET, str(project_id),
                        st_ticket['key'], list(),
                        created_at=st_ticket['issue_created_at'], updated_at=st_ticket['issue_updated_at'],
                        **st_ticket)
                self.add_result(new_st_ticket_item, list())

        current_start_date = '%d-%d-%d' % (datetime.datetime.now().year,
                                           datetime.datetime.now().month, datetime.datetime.now().day)
        self.save_auxiliary({'start_date': current_start_date})

        self.save_result()
