import logging
import datetime

from sandbox.projects.SecDis.data_types import Item, ItemType
from sandbox.projects.SecDis.Collectors import BaseCollector
from sandbox.sdk2.parameters import JSON
from sandbox.sandboxsdk.environments import PipEnvironment


class StTicketRelCollector(BaseCollector):
    """ Service Security Discovery: Monitoring Releases from Startrack Queue """
    collector_name = 'st_ticket_release'
    input_types = (ItemType.QUEUE)
    output_types = (ItemType.RELEASE, ItemType.ST_TICKET)

    class Parameters(BaseCollector.Parameters):
        st_filter_options = JSON("filter_options", default={'resolution':'notEmpty()'}, required=False)
        subissues_queues = JSON("subissues_queues", default=[], required=False)

    class Requirements(BaseCollector.Requirements):
        environments = [PipEnvironment('startrek_client')]

    def get_releases_by_queue(self, queue_name, updated_from, rel_tikets_limit=300, filter_options={}, filter_subissues_queue=[]):
        from startrek_client import Startrek
        from startrek_client.exceptions import Forbidden

        access_token = self.get_vault('OAuthSecDis')
        client = Startrek(useragent="AppSec Discovery Release Collector", base_url='https://st-api.yandex-team.ru',
                          token=access_token)
        issues_filter = {'queue': queue_name, 'updated': {'from': updated_from}}
        for (filter_key, filter_value) in filter_options.items():
            issues_filter[filter_key] = filter_value
        issues = client.issues.find(
            filter=issues_filter,
            order=['-created']
        )
        releases = []
        for index, issue in enumerate(issues):
            if index >= rel_tikets_limit:
                break
            # get information about Conductor components
            issue_conductor_releases = set()
            issue_conductor_components = []
            for remotelink in issue.remotelinks:
                if remotelink.object.application.type == 'ru.yandex.conductor':
                    if remotelink.object.summary.startswith('[stable]') \
                            and remotelink.object.status['name'] == 'done':
                        conductor_releases = sorted(map(lambda s: tuple(s.strip().split('=')),
                                    remotelink.object.summary[len('[stable]'):].split(',')))
                        issue_conductor_releases.add(
                            ', '.join([name+'='+version for (name, version) in conductor_releases]))
                        for (name, version) in conductor_releases:
                            issue_conductor_components.append({'name': name,
                                                            'version': version})
            # get information from Subtickets
            sub_issues = []
            for link in issue.links:
                try:
                    sub_issue_queue = link.object.key.split('-')[0]
                    if len(filter_subissues_queue) > 0 and sub_issue_queue not in filter_subissues_queue:
                        continue
                    sub_issue = {
                        'key': link.object.key,
                        'summary': link.object.summary,
                        'issue_created_at': link.object.createdAt,
                        'issue_updated_at': link.object.updatedAt,
                        'commits': []}
                    for remotelink in link.object.remotelinks:
                        if remotelink.object.application.type == 'com.github':
                            # Pull-requests for github
                            (github_type, github_link) = remotelink.object.key.split(':')
                            if github_type == 'pull':
                                github_link_parts = github_link.split('/')
                                github_link_parts.insert(-1, 'pull')
                                if remotelink.object.application.id == 'ru.yandex.github.enterprise':
                                    github_link_parts.insert(0, 'https://github.yandex-team.ru')
                                elif remotelink.object.application.id == 'com.github':
                                    github_link_parts.insert(0, 'https://github.com')
                                github_link_parts.append('files')
                                github_link = '/'.join(github_link_parts)
                            sub_issue['commits'].append(github_link)
                        elif remotelink.object.application.type == 'ru.yandex.stash':
                            # Pull-requests for bb.yandex-team.ru
                            (bb_project, bb_repo, pr_num) = remotelink.object.key.split('/')
                            sub_issue['commits'].append('/'.join(
                                ['https://bb.yandex-team.ru', 'projects', bb_project,
                                'repos', bb_repo, 'pull-requests', pr_num, 'diff']))
                    sub_issues.append(sub_issue)
                except Forbidden:
                    sub_issue = {'key': link.object.key,
                        'summary': 'Forbidden',
                        'issue_created_at': str(datetime.datetime.now()),
                        'issue_updated_at': str(datetime.datetime.now()),
                        'commits': []
                    }
                    sub_issues.append(sub_issue)
                    continue
            if len(sub_issues) > 0:
                releases.append({
                    'key': issue['key'],
                    'summary': issue['summary'],
                    'release_created_at': issue['createdAt'],
                    'release_updated_at': issue['updatedAt'],
                    'conductor_releases': list(issue_conductor_releases),
                    'conductor_components': list(issue_conductor_components),
                    'sub_issues': sub_issues
                })
        return releases

    def on_execute(self):
        project_id = self.Parameters.project_id
        auxiliary = self.load_auxiliary()
        logging.info("auxiliary %s" % auxiliary)
        if auxiliary is None:
            auxiliary = dict()
        default_start_datetime = datetime.datetime.now() - datetime.timedelta(days=21)
        default_start_date = '%d-%d-%d' % (default_start_datetime.year,
                                           default_start_datetime.month, default_start_datetime.day)
        start_date = auxiliary.get('start_date', default_start_date)

        queue_list = [Item.unserialize(item) for item in self.Parameters.item_list]

        for queue in queue_list:
            releases = self.get_releases_by_queue(queue.get_value(), start_date,\
                filter_options=self.Parameters.st_filter_options, filter_subissues_queue=self.Parameters.subissues_queues)
            for release in releases:
                # Release
                new_release_item = Item(ItemType.RELEASE, str(project_id),
                            release['key'], list(),
                            created_at=release['release_created_at'], updated_at=release['release_updated_at'],
                            **release)
                # sub tickets for Release
                sub_issues_uuids = []
                for st_ticket in release['sub_issues']:
                    new_st_ticket_item = Item(ItemType.ST_TICKET, str(project_id),
                            st_ticket['key'], new_release_item.get_id(),
                            created_at=st_ticket['issue_created_at'], updated_at=st_ticket['issue_updated_at'],
                            **st_ticket)
                    sub_issues_uuids.append(new_st_ticket_item.get_id())
                    self.add_result(new_st_ticket_item, list())
                new_release_item.fields['sub_issues_uuids'] = sub_issues_uuids

                self.add_result(new_release_item, list())

        current_start_date = '%d-%d-%d' % (datetime.datetime.now().year,
                                           datetime.datetime.now().month, datetime.datetime.now().day)
        self.save_auxiliary({'start_date': current_start_date})

        self.save_result()
