import logging
import datetime

from sandbox.projects.SecDis.data_types import Item, ItemType
from sandbox.projects.SecDis.Collectors import BaseCollector

from sandbox.sandboxsdk.environments import PipEnvironment


class TaxiRelCollector(BaseCollector):
    """ Service Security Discovery: Taxi Releases Startrack Queue Monitoring """
    collector_name = 'taxirel'
    input_types = ()
    output_types = (ItemType.RELEASE,)

    class Requirements(BaseCollector.Requirements):
        environments = [PipEnvironment('startrek_client')]

    def get_releases_by_queue(self, queue_name, created_from):
        from startrek_client import Startrek
        from startrek_client.exceptions import Forbidden

        access_token = self.get_vault('OAuthSecDis')
        client = Startrek(useragent="Taxi Release Collector", base_url='https://st-api.yandex-team.ru',
                          token=access_token)
        issues = client.issues.find(
            filter={'queue': queue_name, 'updated': {'from': created_from}, 'resolution': 'notEmpty()'},
            order=['-created']
        )
        releases = []
        for index, issue in enumerate(issues):
            if index > 300:
                break
            # Get Conductor ComponentName and Version
            issue_conductor_releases = []
            for remotelink in issue.remotelinks:
                if remotelink.object.application.type == 'ru.yandex.conductor':
                    if remotelink.object.summary.startswith('[stable]') \
                            and remotelink.object.status['name'] == 'done':
                        conductor_releases = sorted(map(lambda s: tuple(s.strip().split('=')),
                                    remotelink.object.summary[len('[stable]'):].split(',')))
                        issue_conductor_releases.append({
                            'release_name': ', '.join([name+'='+version for (name, version) in conductor_releases]),
                            'components': [{'name': release_name,
                                            'version': release_version}
                                        for (release_name, release_version) in conductor_releases]
                        })
            # Get All SubIssues and Corresponding Commits
            sub_issues = []
            for link in issue.links:
                try:
                    sub_issue = {'key': link.object.key, 'summary': link.object.summary, 'commits': []}
                    for remotelink in link.object.remotelinks:
                        if remotelink.object.application.type == 'com.github':
                            (github_type, github_link) = remotelink.object.key.split(':')
                            if github_type == 'pull':
                                github_link_parts = github_link.split('/')
                                github_link_parts.insert(-1, 'pull')
                                github_link_parts.insert(0, 'https://github.yandex-team.ru')
                                github_link = '/'.join(github_link_parts)
                            sub_issue['commits'].append(github_link)
                    sub_issues.append(sub_issue)
                except Forbidden:
                    sub_issue = {'key': link.object.key, 'summary': 'Forbidden', 'commits': []}
                    sub_issues.append(sub_issue)
                    continue
            for issue_conductor_release in issue_conductor_releases:
                releases.append({
                    'release_name': issue_conductor_release['release_name'],
                    'components': issue_conductor_release['components'],
                    'key': issue['key'],
                    'summary': issue['summary'],
                    'release_created_at': issue['createdAt'],
                    'release_updated_at': issue['updatedAt'],
                    'sub_issues': sub_issues
                })
        return releases

    def on_execute(self):
        project_id = self.Parameters.project_id
        auxiliary = self.load_auxiliary()
        logging.info("auxiliary %s" % auxiliary)
        if auxiliary is None:
            auxiliary = dict()
        default_start_datetime = datetime.datetime.now() - datetime.timedelta(days=7)
        default_start_date = '%d-%d-%d' % (default_start_datetime.year,
                                           default_start_datetime.month, default_start_datetime.day)
        start_date = auxiliary.get('start_date', default_start_date)

        releases = self.get_releases_by_queue('TAXIREL', start_date)
        for release in releases:
            new_item = Item(ItemType.RELEASE, str(project_id),
                        release['release_name'], list(),
                        created_at=release['release_created_at'], updated_at=release['release_updated_at'],
                        **release)
            self.add_result(new_item, list())

        current_start_date = '%d-%d-%d' % (datetime.datetime.now().year,
                                           datetime.datetime.now().month, datetime.datetime.now().day)
        self.save_auxiliary({'start_date': current_start_date})

        self.save_result()
