# -*- coding: utf-8 -*-

import logging
from requests import get
from datetime import datetime
from time import time
import re

from sandbox.sdk2.parameters import List
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.SecDis.data_types import Item, ItemType
from sandbox.projects.SecDis.Collectors import BaseCollector

from sandbox.projects.SecDis.Collectors.utils import Dogma


class TsumCollector(BaseCollector):
    """ Service Security Discovery: Market TSUM monitoring """
    collector_name = 'tsum'
    input_types = ()
    output_types = (ItemType.RELEASE,)

    class Parameters(BaseCollector.Parameters):
        filter = List("Filtered project ids", required=False, default=None)

    class Requirements(BaseCollector.Requirements):
        environments = (
            PipEnvironment('yandex-tracker-client', '1.3', custom_parameters=['--upgrade-strategy only-if-needed']),
            PipEnvironment('startrek-client', '2.3', custom_parameters=['--upgrade-strategy only-if-needed']),
        )

    def __get_startrek_client(self):
        import startrek_client
        st_url = 'https://st-api.yandex-team.ru/v2'
        token = self.get_vault('OAuthSecDis')

        self.st = startrek_client.Startrek(token=token, useragent='', base_url=st_url)

    def get_projects_list(self):
        url = "https://tsum.yandex-team.ru/api/projects"
        headers = {
            'Cookie': self.cookie,
            'Content-Type': 'application/json',
        }
        res = get(url, headers=headers, verify=False)
        if res.status_code != 200:
            logging.error("[get_projects_list] status code %d" % res.status_code)
            return list()
        json_data = res.json()

        return json_data

    def get_project(self, tsum_project_id):
        url = "https://tsum.yandex-team.ru/api/projects/{}".format(tsum_project_id)
        headers = {
            'Cookie': self.cookie,
            'Content-Type': 'application/json',
        }
        res = get(url, headers=headers, verify=False)
        if res.status_code != 200:
            logging.error("[get_project] status code %d" % res.status_code)
            return dict()
        json_data = res.json()

        return json_data

    def get_releases(self, tsum_project_id):
        current_ts = int(time())
        url = "https://tsum.yandex-team.ru/api/projects/{}/completed-releases?" \
              "fromDate={}&toDate={}&page=1&pageSize=100".format(tsum_project_id, current_ts-30*24*3600, current_ts)
        headers = {
            'Cookie': self.cookie,
            'Content-Type': 'application/json',
        }
        res = get(url, headers=headers, verify=False)
        if res.status_code != 200:
            logging.error("[get_releases] status code %d" % res.status_code)
            return list()
        json_data = res.json()

        return json_data.get('releases', [])

    def get_issues(self, key):
        token = self.get_vault('OAuthSecDis')
        dogma_url = 'https://dogma.yandex-team.ru/api/v3'
        dogma = Dogma(dogma_url, token)

        res = {
            'key': key,
        }

        client = self.st
        try:
            issue = client.issues[key]
        except:
            return res
        res['summary'] = issue.summary
        comments = issue.comments.get_all()
        subissues = list()
        for comment in comments:
            author = comment.createdBy.login
            text = comment.text
            if author == 'robot-market-infra' and u'Тикеты, попавшие в релиз' in text:
                subissues = list()
                for t in text.split('\n'):
                    if t.startswith('*'):
                        subkey = t[1:].strip()
                        subissue_dict = {
                            'key': subkey
                        }
                        subissues.append(subissue_dict)
            elif author == 'robot-market-infra' and u'успешно влиты следующие пулл-реквесты' in text:
                subissues = list()
                for t in text.split('\n'):
                    if t.startswith('* (('):
                        t = t[4:]
                        pull, subkey = t.split(' ', 1)
                        m = re.search(r'^[a-zA-Z]+\-\d+', subkey.strip())
                        if m:
                            subkey = m.group(0)
                        else:
                            continue
                        subissue_dict = {
                            'key': subkey,
                            'pull': pull
                        }
                        subissues.append(subissue_dict)
            elif author == 'robot-market-infra' and u'Изменения:' in text:
                subissues = set()
                for t in text.split('\n'):
                    if t.startswith('*'):
                        t = t[2:]
                        subkey, _ = t.split(' ', 1)
                        m = re.search(r'^[a-zA-Z]+\-\d+', subkey.strip())
                        if m:
                            subkey = m.group(0)
                        else:
                            continue
                        subissues.add(subkey)
                subissues = [dict(key=subkey) for subkey in subissues]

        if not subissues and issue.fixVersions:
            fixversion = issue.fixVersions[0].id
            queue = issue.queue.key
            for subissue in client.issues.find('Queue: {} "Fix Version": {}'.format(queue, fixversion)):
                if subissue.key != key and subissue.status.key == 'closed':
                    subissue_dict = {
                        'key': subissue.key
                    }
                    subissues.append(subissue_dict)

        for subissue_dict in subissues:
            try:
                issue = client.issues[subissue_dict['key']]
            except:
                continue
            subissue_dict['summary'] = issue.summary
            subissue_dict['commits'] = list()

            for link in issue.remotelinks:
                application = link.object.application.id
                link_key = link.object.key.split(':', 1)
                if application == 'ru.yandex.github.enterprise' and link_key[0] == 'pull':
                    repo = link_key[1].rsplit('/', 1)[0]
                    pull_id = link_key[1].rsplit('/', 1)[1]
                    pull_request = 'https://github.yandex-team.ru/' + repo + '/pull/' + pull_id
                    subissue_dict['commits'].append(pull_request)

            if subissue_dict['commits']:
                continue

            commits_info = dogma.get_ticket_commits(subissue_dict['key'])
            for commit in commits_info:
                if '_dogma' in commit and 'html_url' in commit['_dogma']:
                    html_url = commit['_dogma']['html_url']
                    subissue_dict['commits'].append(html_url)
        res['sub_issues'] = subissues

        return res

    def get_pipelaunch_info(self, ids):
        d = {}
        if not ids:
            return d
        pipe_id = ids[0]
        url = "https://tsum.yandex-team.ru/api/pipe/launch/{}".format(pipe_id)
        headers = {
            'Cookie': self.cookie,
            'Content-Type': 'application/json',
        }
        res = get(url, headers=headers, verify=False)
        if res.status_code != 200:
            logging.error("[get_pipeline_info] status code %d" % res.status_code)
            return d
        json_data = res.json()
        jobs = json_data.get('jobs', [])
        for job in jobs[:2]:
            task_states = job.get('taskStates', [])
            if not task_states:
                continue
            task = task_states[0]
            module = task.get('module', '')
            url = task.get('url', '')
            if module == 'STARTREK':
                key = url.rsplit('/')[-1]
                d.update(self.get_issues(key))
            elif module == 'GITHUB':
                d['github'] = url

        return d

    def get_pipeline_titles(self, tsum_project_id):
        d = {}
        url = "https://tsum.yandex-team.ru/api/projects/{}".format(tsum_project_id)
        headers = {
            'Cookie': self.cookie,
            'Content-Type': 'application/json',
        }
        res = get(url, headers=headers, verify=False)
        if res.status_code != 200:
            logging.error("[get_pipeline_titles] status code %d" % res.status_code)
            return d
        json_data = res.json()
        pipelines = json_data.get('pipelines', [])
        for _id, data in pipelines.items():
            d[_id] = data.get('title', _id)

        return d

    def on_execute(self):
        self.__get_startrek_client()

        self.cookie = self.get_vault('TSUMCookieSecDis')
        auxiliary = self.load_auxiliary()
        logging.info("auxiliary %s" % auxiliary)
        if auxiliary is None:
            auxiliary = dict()
        projects_ids = auxiliary.get('projects_ids', list())
        releases_ids = auxiliary.get('releases_ids', list())

        project_id = self.Parameters.project_id

        projects_list = self.get_projects_list()
        for project in projects_list:
            if project['id'] not in projects_ids:
                projects_ids.append(project['id'])
            tsum_project_id = project['id']
            tsum_pipe_title_dict = self.get_pipeline_titles(tsum_project_id)
            tsum_project_title = project['title']

            if self.Parameters.filter and tsum_project_id not in self.Parameters.filter:
                continue

            releases_list = self.get_releases(tsum_project_id)
            for release in releases_list:
                if release['id'] not in releases_ids and release['state'] == 'FINISHED':
                    releases_ids.append(release['id'])

                    pipe_launch_ids = release.get('pipeLaunchIds', [])
                    pipelaunch_info = self.get_pipelaunch_info(pipe_launch_ids)

                    created_at = datetime.fromtimestamp(release['createdDate']/1000).strftime('%Y-%m-%dT%H:%M:%SZ')
                    updated_at = created_at

                    value = u'{} {} {}'.format(tsum_project_title,
                                                  tsum_pipe_title_dict.get(release['pipeId'], release['pipeId']),
                                                  release['title'])
                    new_item = Item(ItemType.RELEASE, project_id, value, list(), created_at, updated_at,
                                    release_id=release['id'], title=release['title'],
                                    pipe_id=release['pipeId'], pipe_title=tsum_pipe_title_dict.get(release['pipeId'], ''),
                                    tsum_project_id=tsum_project_id, tsum_project_title=tsum_project_title,
                                    created_ts=release['createdDate'], triggered_by=release['triggeredBy'], **pipelaunch_info)
                    self.add_result(new_item, list())

        auxiliary = {
            'projects_ids': projects_ids,
            'releases_ids': releases_ids
        }
        self.save_auxiliary(auxiliary)

        self.save_result()
