import logging
from requests import get

from sandbox.projects.SecDis.data_types import Item, ItemType
from sandbox.projects.SecDis.Collectors import BaseCollector


class YtCollector(BaseCollector):
    """ Service Security Discovery: yt tables by ABC persons collector task """
    collector_name = 'yt'
    input_types = (ItemType.PERSON,)
    output_types = (ItemType.RESOURCE,)

    def _get_yt_accounts(self, cluster):
        data = {
            'path': '//sys/accounts',
            'attributes': ['responsibles', 'owner', 'tables'],
            'max_size': 20000,
        }
        headers = {
            'Authorization': 'OAuth {}'.format(self.yt_access_token),
            'Accept': 'application/json',
        }
        url = 'https://{}.yt.yandex-team.ru/api/v3/list'.format(cluster)
        res = get(url, headers=headers, params=data)
        json_data = res.json()

        return json_data

    def _get_yt_tables(self, cluster, path, account_dict, depth=2):
        def parse_acl(acl_list):
            result = {
                'read': [],
                'write': [],
            }
            for acl in acl_list:
                if 'read' in acl['permissions']:
                    result['read'] += acl['subjects']
                if 'write' in acl['permissions']:
                    result['write'] += acl['subjects']
            return result

        if depth == 0:
            return
        data = {
            'path': path,
            'attributes': ['type', 'account', 'effective_acl', 'path'],
            'max_size': 20000,
        }
        headers = {
            'Authorization': 'OAuth {}'.format(self.yt_access_token),
            'Accept': 'application/json',
        }
        url = 'https://{}.yt.yandex-team.ru/api/v3/list'.format(cluster)
        res = get(url, headers=headers, params=data)
        json_data = res.json()

        path_added = False
        for node in json_data:
            if type(node) is not dict:
                continue
            attributes = node['$attributes']
            name = node['$value']
            node_type = attributes['type']
            account = attributes['account']
            node_path = attributes['path']
            num = len(node_path.split('/')) - 2

            if account in account_dict:
                if node_type == 'table' or node_type == 'map_node' and name in ['5min', '30min', '1h', '1d']:
                    if path_added:
                        continue
                    logging.info('table: {} {}'.format(cluster, path.encode()))
                    new_item = Item(ItemType.RESOURCE, self.Parameters.project_id, path, account_dict[account],
                                    cluster=cluster, resource_type='yt', acl=parse_acl(attributes['effective_acl']))
                    self.add_result(new_item, account_dict[account])
                    path_added = True
                elif num > 3:
                    logging.info('table: {} {}'.format(cluster, node_path.encode()))
                    new_item = Item(ItemType.RESOURCE, self.Parameters.project_id, node_path, account_dict[account],
                                    cluster=cluster, resource_type='yt', acl=parse_acl(attributes['effective_acl']))
                    self.add_result(new_item, account_dict[account])
                else:
                    self._get_yt_tables(cluster, node_path, account_dict, depth)
            else:
                self._get_yt_tables(cluster, node_path, account_dict, depth-1)

    def _get_yt_resources(self, item_list, cluster):
        self.yt_access_token = self.get_vault('OAuthYTSecDis')

        yt_accounts = self._get_yt_accounts(cluster)
        person_dict = dict()
        for item in item_list:
            person_dict[item.get_value()] = item.get_id()
        account_dict = dict()

        for value in yt_accounts:
            if type(value) is not dict:
                continue
            account = value['$value']
            responsibles = set(value['$attributes'].get('responsibles', list()))
            if responsibles and responsibles.issubset(person_dict.keys()):
                account_dict[account] = [person_dict[r] for r in responsibles]
        logging.info('account_dict:\n{}'.format(account_dict))

        self._get_yt_tables(cluster, '//home', account_dict)

    def get_yt_resources(self, item_list):
        self._get_yt_resources(item_list, 'hahn')
        self._get_yt_resources(item_list, 'banach')

    def on_execute(self):
        person_list = [Item.unserialize(item) for item in self.Parameters.item_list if item['type'] == ItemType.PERSON]
        self.get_yt_resources(person_list)

        self.save_result()
