import logging
from sandbox import sdk2
from sandbox.common.types.client import Tag
from json import loads, dumps
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.ssh import Key
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk import svn

from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.SecDis import BaseResource


class BaseCollector(sdk2.Task):
    """ Service Security Discovery base collector class """
    collector_name = 'base'
    input_types = ()
    output_types = ()

    class Requirements(sdk2.Requirements):
        environments = [PipEnvironment('git+git://github.yandex-team.ru/product-security/yaseclib.git')]
        disk_space = 1024
        ram = 1024
        client_tags = Tag.Group.LINUX

    class Context(sdk2.Context):
        pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 21600
        project_id = sdk2.parameters.String("Project Id", required=True)
        item_list = sdk2.parameters.JSON("List of events", required=False)
        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Output') as config_block:
                result = sdk2.parameters.JSON('Collector output')

    def get_vault(self, name):
        vault = sdk2.Vault.data(name)
        return vault

    def save_auxiliary(self, data):
        auxiliary_path = 'auxiliary_{}_{}.json'.format(self.Parameters.project_id, self.collector_name)
        logging.info("Creating new resource file %s" % auxiliary_path)
        auxiliary_resource = sdk2.ResourceData(BaseResource(self, "Output file", auxiliary_path))
        data = dumps(data)
        auxiliary_resource.path.write_bytes(data)
        auxiliary_resource.ready()

    def load_auxiliary(self):
        batch_size = 100
        resources = sdk2.Resource.find(BaseResource).limit(batch_size)

        auxiliary_path = 'auxiliary_{}_{}.json'.format(self.Parameters.project_id, self.collector_name)
        auxiliary_resource = None

        for resource in resources:
            path = str(resource.path)
            state = resource.state
            created = resource.created
            logging.info("Found: resource %s state %s created %s" % (path, state, created))
            if state != 'READY':
                continue
            if path == auxiliary_path:
                auxiliary_resource = sdk2.ResourceData(resource)
                break

        if auxiliary_resource is None:
            return None

        data = auxiliary_resource.path.read_bytes()
        data = loads(data)

        return data

    def add_result(self, item, ancestor_id):
        if getattr(self, 'result_dict', None) is None:
            self.result_dict = dict()

        item_id = item.get_id()
        if item_id in self.result_dict:
            self.result_dict[item_id].append_ancestor(ancestor_id)
        else:
            self.result_dict[item_id] = item

    def save_result(self):
        items = [item.serialize() for item in getattr(self, 'result_dict', dict()).values()]
        self.Parameters.result = items

class CloneRepositoryCollector(BaseCollector):
    """ Service Security Discovery: Collector with clone repository functions """
    def _git_clone(self, repo, branch=None):
        repo = repo.split('/', 3)[-1]
        folder = str(self.path('work'))
        if branch is None:
            branch_arg = ''
        else:
            branch_arg = '-b {branch}'.format(branch=branch)
        git_cmd = 'ssh-keyscan -H github.yandex-team.ru >> ~/.ssh/known_hosts && ' \
                  'git clone --depth 1 git@github.yandex-team.ru:{repo}.git ' \
                  '{branch_arg} {folder}'.format(repo=repo, branch_arg=branch_arg, folder=folder)

        try:
            with Key(self, self.owner, 'STOP_LEAK_SSH_KEY'), ProcessLog(self, logger=logging.getLogger('git')) as pl:
                sp.check_call(git_cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout)
            return folder
        except sp.CalledProcessError as e:
            logging.error('Run git clone error: %s', e)
            return None

    def _bb_clone(self, repo, branch=None):
        repo = repo.split('/', 3)[-1]
        folder = str(self.path('work'))
        if branch is None:
            branch_arg = ''
        else:
            branch_arg = '-b {branch}'.format(branch=branch)
        git_cmd = 'git clone --depth 1 https://{username}:{token}@bb.yandex-team.ru/scm/{repo}.git ' \
                    ' {branch_arg} {folder} '.format(username=self.get_vault('SECDIS_BB_USERNAME'), \
                    token=self.get_vault('SECDIS_BB_TOKEN'), repo=repo, branch_arg=branch_arg, folder=folder)
        with ProcessLog(self, logger=logging.getLogger('git')) as pl:
            sp.check_call(git_cmd, shell=True, stderr=pl.stdout, stdout=pl.stdout)
        return folder

    def _arcadia_clone(self, repo):
        folder = str(self.path('work'))
        checkout_arcadia_from_url = 'arcadia:/arc/trunk/arcadia'
        revision = svn.Arcadia.parse_url(checkout_arcadia_from_url).revision
        svn.Arcadia.checkout(repo, folder, revision=revision)
        return folder

    def clone(self, item):
        repo = item.get_value()
        branch = item.fields.get('branch')

        if repo.startswith('https://github.yandex-team.ru/'):
            folder = self._git_clone(repo, branch)
        elif repo.startswith('https://bb.yandex-team.ru/'):
            folder = self._bb_clone(repo, branch)
        elif repo.startswith('https://a.yandex-team.ru/'):
            folder = self._arcadia_clone(repo)
        else:
            folder = None

        return folder
