from requests import get
from time import strptime, mktime


class Dogma(object):
    base_url = ''
    token = ''

    def __init__(self, base_url, token):
        self.base_url = base_url
        self.token = token
        self.headers = {
            'Authorization': 'OAuth ' + self.token
            }

    def get_repositories_by_queue(self, queue, until_ts=0):
        """
        Get all repositories related to queue by reading all attached commits from dogma.
        If use with until_ts param - read only newer attached commits.
        """

        repos = set()
        page = 1
        running = True

        while running:
            url = "{}/search/commits?q=queues:{}+exclude_dublicates:true+parents:false&numpages=true" \
                  "&per_page=200&page={}".format(self.base_url, queue, page)
            r = get(url, headers=self.headers)
            if r.status_code != 200:
                break
            commits = r.json()

            for commit in commits:
                if until_ts:
                    date_str = commit['commit']['author']['date']
                    date = strptime(date_str, "%Y-%m-%dT%H:%M:%S")
                    ts = int(mktime(date))
                    if ts < until_ts:
                        running = False
                        break

                repo = commit.get('repo', {})
                url = repo.get('url')
                if url:
                    repos.add(url)

            if not r.headers.get('Link'):
                break
            page += 1

        return list(repos)

    def get_last_commits(self, queue, count=0):
        """
        Get last commits related to queue.
        If use with count=0 - get all commits.
        """

        commits = list()
        page = 1
        read_count = 0

        while True:
            if count:
                per_page = min(count-read_count, 200)
            else:
                per_page = 200
            url = "{}/search/commits?q=queues:{}+exclude_dublicates:true+parents:false&numpages=true" \
                  "&per_page={}&page={}".format(self.base_url, queue, per_page, page)
            r = get(url, headers=self.headers, verify=False)
            if r.status_code != 200:
                break
            json_data = r.json()
            read_count += len(json_data)
            commits += json_data

            if read_count == count:
                break
            if not r.headers.get('Link'):
                break
            page += 1

        return commits

    def get_ticket_commits(self, key):
        """
        Get commits related to startrek ticket.
        """

        commits = list()
        page = 1
        per_page = 200

        while True:
            url = "{}/search/commits?q=tickets:{}+exclude_dublicates:true+parents:false&numpages=true" \
                  "&per_page={}&page={}".format(self.base_url, key, per_page, page)
            r = get(url, headers=self.headers, verify=False)
            if r.status_code != 200:
                break
            json_data = r.json()
            commits += json_data

            if not r.headers.get('Link'):
                break
            page += 1

        return commits


class Abc(object):
    base_url = ''
    token = ''

    def __init__(self, base_url, token):
        self.base_url = base_url
        self.token = token
        self.headers = {
            'content-type': 'application/json',
            'Authorization': 'OAuth ' + self.token
            }

    def get_service_by_slug(self, slug):
        payload = {
                'slug': slug,
                'fields': 'id,created_at,modified_at,name,slug'
                }
        url = self.base_url + '/services/'
        r = get(url, params=payload, headers=self.headers)
        json_data = r.json()
        return json_data['results'][0]

    def get_service_by_id(self, abc_id):
        url = '%s%s%s%s' % (self.base_url, '/services/', abc_id, '/?fields=id,created_at,modified_at,name,slug')
        r = get(url, headers=self.headers)
        json_data = r.json()
        return json_data

    def get_service_by_id_with_descendants(self, abc_id):
        service = self.get_service_by_id(abc_id)
        result = [service]
        
        url = self.base_url + '/services/'+'?parent__with_descendants='+str(abc_id)+'&fields=id,created_at,modified_at,name,slug'
        while url:
            r = get(url, headers=self.headers)
            json_data = r.json()
            if json_data and 'error' not in json_data:
                result += json_data['results']
                url = json_data.get('next', None)
            else:
                url = None
        return result

    def get_service_members_by_id(self, abc_id, with_descendants=False):
        result = []
        if with_descendants:
            key = 'service__with_descendants'
        else:
            key = 'service'
        url = self.base_url + '/services/members/?'+str(key)+'='+str(abc_id)+'&fields=id,person,created_at,modified_at'
        while url:
            r = get(url, headers=self.headers)
            json_data = r.json()
            if json_data and 'error' not in json_data:
                result += json_data['results']
                url = json_data.get('next', None)
            else:
                url = None
        return result

    def get_service_resources_by_id(self, abc_id, types=list(), with_childs=False):
        result = []
        payload = {
            'service': abc_id,
            'with_childs': with_childs,
            'type': types,
            'fields': 'resource,state,id,created_at,modified_at',
        }
        url = self.base_url + '/resources/consumers/'
        r = get(url, params=payload, headers=self.headers)
        if json_data and 'error' not in json_data:
            result += json_data['results']
            url = json_data.get('next', None)
        else:
            url = None

        while url:
            r = get(url, headers=self.headers)
            json_data = r.json()
            if json_data and 'error' not in json_data:
                result += json_data['results']
                url = json_data.get('next', None)
            else:
                url = None
        return result


class Startrek(object):
    base_url = ''
    token = ''

    def __init__(self, base_url, token):
        self.base_url = base_url
        self.token = token
        self.headers = {
            'content-type': 'application/json',
            'Authorization': 'OAuth ' + self.token
            }

    def get_queues(self):
        queues = list()
        page = 1
        payload = {
                'perPage': 100,
                }
        url = self.base_url + '/queues/'
        while True:
            payload['page'] = page
            r = get(url, params=payload, headers=self.headers)
            if r.status_code != 200:
                break
            json_data = r.json()
            if not json_data:
                break
            queues += json_data
            page += 1

        return queues

    def get_queue_components(self, queue):
        url = self.base_url + 'queues/{}/components'.format(queue)
        r = get(url, headers=self.headers)
        if r.status_code != 200:
            return []
        json_data = r.json()

        return json_data


class Conductor(object):
    base_url = ''

    def __init__(self, base_url='https://c.yandex-team.ru/api/'):
        self.base_url = base_url

    def get_hosts_for_project(self, project_name):
        r = get(self.base_url + 'projects2hosts/' + project_name)
        if r.status_code == 200:
            return r.content.strip().split("\n")
        else:
            return []

    def get_tasks_for_project(self, project_name, branch='stable', limit=1000):
        r = get(self.base_url + 'tasks_filter?project=' + project_name + '&branch=' + branch + '&limit=' + str(limit))
        if r.status_code == 200:
            return r.json()
        else:
            return []
