import logging
from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.common.errors import TaskFailure
import sandbox.common.types.task as ctt
from json import loads, dumps
from requests import post

from sandbox.projects.SecDis import BaseResource

from sandbox.projects.SecDis.Collectors.RepoCollector import RepoCollector
from sandbox.projects.SecDis.Collectors.QueueCollector import QueueCollector
from sandbox.projects.SecDis.Collectors.AbcCollector import AbcCollector
from sandbox.projects.SecDis.Collectors.AbcResourceCollector import AbcResourceCollector
from sandbox.projects.SecDis.Collectors.CommitCollector import CommitCollector
from sandbox.projects.SecDis.Collectors.TsumCollector import TsumCollector
from sandbox.projects.SecDis.Collectors.TaxiRelCollector import TaxiRelCollector
from sandbox.projects.SecDis.Collectors.ConductorReleaseCollector import ConductorReleaseCollector
from sandbox.projects.SecDis.Collectors.HandleCollector import HandleCollector
from sandbox.projects.SecDis.Collectors.HandleYTCollector import HandleYTCollector
from sandbox.projects.SecDis.Collectors.YtCollector import YtCollector
from sandbox.projects.SecDis.Collectors.MacrosCollector import MacrosCollector
from sandbox.projects.SecDis.Collectors.StTicketRelCollector import StTicketRelCollector
from sandbox.projects.SecDis.Collectors.StTicketCollector import StTicketCollector
from sandbox.projects.SecDis.Collectors.MailRelCollector import MailRelCollector
from sandbox.projects.SecDis.Collectors.MicroserviceFromCodeCollector import MicroserviceFromCodeCollector
from sandbox.projects.SecDis.Collectors.GeoRelCollector import GeoRelCollector

tasks = {
    'REPO_COLLECTOR': RepoCollector,
    'QUEUE_COLLECTOR': QueueCollector,
    'ABC_COLLECTOR': AbcCollector,
    'ABC_RESOURCE_COLLECTOR': AbcResourceCollector,
    'COMMIT_COLLECTOR': CommitCollector,
    'TSUM_COLLECTOR': TsumCollector,
    'TAXIREL_COLLECTOR': TaxiRelCollector,
    'CONDUCTOR_RELEASE_COLLECTOR': ConductorReleaseCollector,
    'HANDLE_COLLECTOR': HandleCollector,
    'HANDLE_YT_COLLECTOR': HandleYTCollector,
    'YT_COLLECTOR': YtCollector,
    'MACROS_COLLECTOR': MacrosCollector,
    'ST_TICKET_REL_COLLECTOR': StTicketRelCollector,
    'ST_TICKET_COLLECTOR': StTicketCollector,
    'MAIL_REL_COLLECTOR': MailRelCollector,
    'MICROSERVICE_FROM_CODE_COLLECTOR': MicroserviceFromCodeCollector,
    'GEO_REL_COLLECTOR': GeoRelCollector
}


class SecDisMeta(sdk2.Task):
    """ Service Security Discovery meta task """

    class Requirements(sdk2.Requirements):
        disk_space = 1024
        ram = 1024
        client_tags = Tag.Group.LINUX

    class Parameters(sdk2.Parameters):
        kill_timeout = 21600
        project_id = sdk2.parameters.String("Project Id", required=True)
        input_data = sdk2.parameters.JSON("Input", required=True)
        pipeline = sdk2.parameters.List("Pipeline", required=True)
        optional = sdk2.parameters.JSON("Optional", required=False, default=dict())

        with sdk2.parameters.Output():
            with sdk2.parameters.Group('Output') as config_block:
                result = sdk2.parameters.JSON('Output')

    class Context(sdk2.Context):
        child_id = None

    def get_vault(self, name):
        vault = sdk2.Vault.data(name)
        return vault

    def save_auxiliary(self, data):
        auxiliary_path = 'auxiliary_{}_{}.json'.format(self.Parameters.project_id, self.collector_type)
        logging.info("Creating new resource file %s" % auxiliary_path)
        auxiliary_resource = sdk2.ResourceData(BaseResource(self, "Output file", auxiliary_path))
        data = dumps(data)
        auxiliary_resource.path.write_bytes(data)
        auxiliary_resource.ready()

    def load_auxiliary(self):
        batch_size = 100
        resources = sdk2.Resource.find(BaseResource).limit(batch_size)

        auxiliary_path = 'auxiliary_{}_{}.json'.format(self.Parameters.project_id, self.collector_type)
        auxiliary_resource = None

        for resource in resources:
            path = str(resource.path)
            state = resource.state
            created = resource.created
            logging.info("Found: resource %s state %s created %s" % (path, state, created))
            if state != 'READY':
                continue
            if path == auxiliary_path:
                auxiliary_resource = sdk2.ResourceData(resource)
                break

        if auxiliary_resource is None:
            return None

        data = auxiliary_resource.path.read_bytes()
        data = loads(data)

        return data

    def save_result(self, data):
        self.Parameters.result = data

    def export_result(self, item_list):
        if not item_list:
            return
        url = 'https://appsec-discovery.sec.yandex-team.ru/api/for_collectors/add'
        data_dict = {self.Parameters.project_id: []}
        for item in item_list:
            if 'project_id' in item:
                if item['project_id'] not in data_dict:
                    data_dict[item['project_id']] = []
                data_dict[item['project_id']].append(item)
            else:
                data_dict[self.Parameters.project_id].append(item)

        data = []
        for project_id in data_dict.keys():
            data.append(dict(project_id=project_id, objects=data_dict[project_id]))
        try:
            post(url, json=data, verify=False)
        except:
            raise TaskFailure("Can't send event data to Qloud API: {}".format(url))

    def on_execute(self):
        pipeline = self.Parameters.pipeline
        pipeline_len = len(self.Parameters.pipeline)
        optional = self.Parameters.optional

        with self.memoize_stage.pipeline_step(pipeline_len) as st:
            collector_num = st.runs - 1
            task_name = pipeline[collector_num]
            logging.info("Entering stage {} collector {}".format(collector_num, task_name))
            collector_cls = tasks.get(task_name)
            if collector_cls is None:
                raise TaskFailure("Wrong collector task name")

            if st.runs == 1:
                item_list = self.Parameters.input_data
            else:
                item_list = sdk2.Task[self.Context.child_id].Parameters.result
                self.export_result(item_list)

            collector_input = list()
            for item in item_list:
                if item['type'] in collector_cls.input_types:
                    collector_input.append(item)

            if collector_cls.input_types and not collector_input:
                raise TaskFailure("Wrong pipeline: no relevant data for collector {}".format(task_name))

            sub_task = collector_cls(
                self,
                description="{} task".format(task_name),
                project_id=self.Parameters.project_id,
                item_list=collector_input,
                **optional
            )

            child_id = sub_task.enqueue().id
            self.Context.child_id = child_id

            raise sdk2.WaitTask(child_id, ctt.Status.Group.FINISH, wait_all=True)

        child_id = self.Context.child_id
        item_list = sdk2.Task[child_id].Parameters.result
        self.export_result(item_list)
        self.save_result(item_list)
